#include <infra/netmon/infra.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/settings.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/json/json_reader.h>

namespace NNetmon {
    namespace {
        const TString SERVICE_ID = "311"; // Netmon
        const TDuration INFRA_REQUEST_WINDOW = TDuration::Minutes(15);
    }

    class TInfraUpdater::TImpl: public TScheduledTask {
    public:
        TImpl(const TTopologyStorage& topologyStorage)
            : TScheduledTask(TDuration::Minutes(1))
            , TopologyStorage(topologyStorage)
        {
        }

        TThreadPool::TFuture Run() override {
            auto now = TInstant::Now();
            auto from = now;
            auto to = now + INFRA_REQUEST_WINDOW;

            TCgiParameters params({
                {"from", ToString(from.Seconds())},
                {"to", ToString(to.Seconds())},
                {"environmentId", TSettings::Get()->GetInfraEnvironmentId()},
                {"serviceId", SERVICE_ID}
            });

            TString url = TSettings::Get()->GetInfraUrl() + "/v1/events?" + params.Print();
            return THttpRequester::Get()->MakeRequest(url).Subscribe([this, url](const THttpRequester::TFuture& future) {
                try {
                    ProcessInfraResponse(future.GetValue());
                } catch (...) {
                    ERROR_LOG << "Request to " << url << " failed: " << CurrentExceptionMessage() << Endl;
                }
            }).IgnoreResult();
        }

        bool IsInfraEventOngoing(const TStringBuf& dc) const {
            auto events = EventsBox.Own();
            if (events->contains(dc)) {
                auto now = TInstant::Now();
                for (const auto& event: events->at(dc)) {
                    if (now >= event.first && now <= event.second) {
                        return true;
                    }
                }
            }
            return false;
        }

    private:
        // event is a pair<start time, finish time>
        using TEvent = std::pair<TInstant, TInstant>;

        void ProcessInfraResponse(const NHttpFetcher::TResultRef& response) {
            const TString& data = response->Data;
            NJson::TJsonValue rootObj;
            NJson::ReadJsonFastTree(data, &rootObj, true);

            THashMap<TString, TVector<TEvent>> events;
            for (const auto& event : rootObj.GetArraySafe()) {
                try {
                    const auto& eventMap = event.GetMapSafe();
                    for (const auto& dc : TopologyStorage.GetTopology()->GetDatacenters()) {
                        const auto& dcName = dc->GetName();
                        const auto& rawDcName = TTopologySettings::Get()->GetRawDc(dcName);

                        if (eventMap.contains(rawDcName) && eventMap.at(rawDcName).GetBooleanSafe()) {
                            auto startTs = eventMap.at("start_time").GetUIntegerSafe();
                            ui64 finishTs = std::numeric_limits<ui64>::max();
                            try {
                                finishTs = eventMap.at("finish_time").GetUIntegerSafe();
                            }
                            catch (...) {}

                            events[dcName].emplace_back(TInstant::Seconds(startTs), TInstant::Seconds(finishTs));
                        }
                    }
                }
                catch (...) {
                    ERROR_LOG << "Can't parse event: " << CurrentExceptionMessage()
                              << " raw event: " << event << Endl;
                }
            }

            if (!events.empty()) {
                INFO_LOG << "Got " << events.size() << " event(s) from infra" << Endl;
            }
            *EventsBox.Own() = std::move(events);
        }

        const TTopologyStorage& TopologyStorage;
        TPlainLockedBox<THashMap<TString, TVector<TEvent>>> EventsBox;
    };

    TInfraUpdater::TInfraUpdater(const TTopologyStorage& topologyStorage)
        : TInfraUpdater(topologyStorage, !TSettings::Get()->GetInfraUrl().empty())
    {
    }

    TInfraUpdater::TInfraUpdater(const TTopologyStorage& topologyStorage, bool schedule)
        : Impl(MakeHolder<TImpl>(topologyStorage))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TInfraUpdater::~TInfraUpdater() = default;

    bool TInfraUpdater::IsInfraEventOngoing(const TStringBuf& dc) const {
        return Impl->IsInfraEventOngoing(dc);
    }

    TThreadPool::TFuture TInfraUpdater::SpinAndWait() noexcept {
        return Impl->SpinAndWait();
    }
}
