#include <infra/netmon/library/api_handler_helpers.h>
#include <infra/netmon/infra.h>
#include <infra/netmon/topology/common_ut.h>
#include <infra/netmon/settings.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

using namespace NNetmon;

namespace {
    struct TInfraContext: public TNonCopyable {
    };

    class TInfraReply: public THttpReply<THttpDispatcher, TInfraContext> {
    public:
        TInfraReply(TInfraContext& serverContext, const TServiceRequest::TRef clientContext)
            : THttpReply<THttpDispatcher, TInfraContext>(serverContext, clientContext)
        {
        }

        void PreprocessRequest(THttpInput&) {
        }
        void ParseRequest(THttpInput&) {
        }

        void WriteResponse(THttpOutput& output) {
            output << THttpResponse()
                .SetContentType(TStringBuf("application/json"))
                .SetContent(Response.Str());
            output.Finish();
        }

    protected:
        NJsonWriter::TBuf Response;
    };

    class TInfraOngoingEventReply: public TInfraReply {
    public:
        using TInfraReply::TInfraReply;

        void Process() {
            auto now = TInstant::Now();
            Response
                .BeginList()
                    .BeginObject() // this event hasn't started yet
                        .WriteKey("start_time")
                        .WriteULongLong((now + TDuration::Minutes(10)).Seconds())
                        .WriteKey("finish_time")
                        .WriteULongLong((now + TDuration::Minutes(30)).Seconds())
                        .WriteKey("man").WriteBool(true)
                        .WriteKey("myt").WriteBool(false)
                        .WriteKey("sas").WriteBool(false)
                        .WriteKey("vla").WriteBool(false)
                        .WriteKey("iva").WriteBool(false)
                    .EndObject()
                    .BeginObject() // this one is ongoing in man
                        .WriteKey("start_time")
                        .WriteULongLong((now - TDuration::Minutes(5)).Seconds())
                        .WriteKey("finish_time")
                        .WriteULongLong((now + TDuration::Minutes(15)).Seconds())
                        .WriteKey("man").WriteBool(true)
                        .WriteKey("myt").WriteBool(false)
                        .WriteKey("sas").WriteBool(false)
                        .WriteKey("vla").WriteBool(false)
                        .WriteKey("iva").WriteBool(false)
                    .EndObject()
                    .BeginObject() // finish_time is null(+inf)
                        .WriteKey("start_time")
                        .WriteULongLong((now - TDuration::Minutes(5)).Seconds())
                        .WriteKey("finish_time")
                        .WriteNull()
                        .WriteKey("man").WriteBool(false)
                        .WriteKey("myt").WriteBool(false)
                        .WriteKey("sas").WriteBool(false)
                        .WriteKey("vla").WriteBool(true)
                        .WriteKey("iva").WriteBool(false)
                    .EndObject()
                .EndList();
        }
    };

    class TInfraOtherDcEventReply: public TInfraReply {
    public:
        using TInfraReply::TInfraReply;

        void Process() {
            auto now = TInstant::Now();
            Response
                .BeginList()
                    .BeginObject() // this event hasn't started yet
                        .WriteKey("start_time")
                        .WriteULongLong((now + TDuration::Minutes(10)).Seconds())
                        .WriteKey("finish_time")
                        .WriteULongLong((now + TDuration::Minutes(30)).Seconds())
                        .WriteKey("man").WriteBool(true)
                        .WriteKey("myt").WriteBool(false)
                        .WriteKey("sas").WriteBool(false)
                        .WriteKey("vla").WriteBool(false)
                        .WriteKey("iva").WriteBool(false)
                    .EndObject()
                    .BeginObject() // this one is ongoing in myt
                        .WriteKey("start_time")
                        .WriteULongLong((now - TDuration::Minutes(5)).Seconds())
                        .WriteKey("finish_time")
                        .WriteULongLong((now + TDuration::Minutes(15)).Seconds())
                        .WriteKey("man").WriteBool(false)
                        .WriteKey("myt").WriteBool(true)
                        .WriteKey("sas").WriteBool(false)
                        .WriteKey("vla").WriteBool(false)
                        .WriteKey("iva").WriteBool(false)
                    .EndObject()
                .EndList();
        }
    };

    class TInfraTest: public TTestBase {
        UNIT_TEST_SUITE(TInfraTest);
        UNIT_TEST(TestOngoingEvent)
        UNIT_TEST(TestOtherDcEvent)
        UNIT_TEST_SUITE_END();

    public:
        void SetUp() override {
            Host = "localhost";
            Port = PortManager.GetPort(12345);
            TSettings::Get()->SetInfraUrl(TStringBuilder() << "http://" << Host << ":" << Port);
        }

        void TestOngoingEvent() {
            TInfraContext context;
            THttpHandler<TInfraOngoingEventReply> infraService(context);
            auto server = TWebServer(THttpServerOptions().SetHost(Host).SetPort(Port).SetThreads(1));
            server.Add("/v1/events", infraService);
            TWebServer::TThreadGuard serverGuard(server);

            TInfraUpdater infra(TGlobalTopology::GetTopologyStorage());
            auto future = infra.SpinAndWait();
            future.Wait();

            UNIT_ASSERT(!future.HasException());
            UNIT_ASSERT(infra.IsInfraEventOngoing("man"));
            UNIT_ASSERT(infra.IsInfraEventOngoing("vla"));
            UNIT_ASSERT(!infra.IsInfraEventOngoing("myt"));
        }

        void TestOtherDcEvent() {
            TInfraContext context;
            THttpHandler<TInfraOtherDcEventReply> infraService(context);
            auto server = TWebServer(THttpServerOptions().SetHost(Host).SetPort(Port).SetThreads(1));
            server.Add("/v1/events", infraService);
            TWebServer::TThreadGuard serverGuard(server);

            TInfraUpdater infra(TGlobalTopology::GetTopologyStorage());
            auto future = infra.SpinAndWait();
            future.Wait();

            UNIT_ASSERT(!future.HasException());
            UNIT_ASSERT(!infra.IsInfraEventOngoing("man"));
            UNIT_ASSERT(infra.IsInfraEventOngoing("myt"));
        }

        TPortManager PortManager;
        ui16 Port;
        TString Host;
    };
}

UNIT_TEST_SUITE_REGISTRATION(TInfraTest);
