#pragma once

#include <infra/netmon/library/requester.h>
#include <infra/netmon/library/thread_pool.h>
#include <infra/netmon/library/settings.h>

#include <library/cpp/neh/rpc.h>
#include <library/cpp/blockcodecs/codecs.h>
#include <library/cpp/blockcodecs/stream.h>

#include <util/stream/mem.h>
#include <util/stream/buffer.h>
#include <util/string/builder.h>

#include <google/protobuf/messagext.h>
#include <google/protobuf/util/json_util.h>
#include <contrib/libs/flatbuffers/include/flatbuffers/flatbuffers.h>

namespace NNetmon {
    namespace {
        const size_t DEFAULT_BUFFER_LENGTH = 10 * 1024 * 1024;

        const NBlockCodecs::ICodec* CODEC = NBlockCodecs::Codec(TStringBuf("snappy"));

        const TDuration REQUEST_TIMEOUT = TDuration::Seconds(30);
    }

    // code 404 Not Found
    class TBadRequest: public yexception {
    };

    template <class T>
    inline void ReadProto(IInputStream* input, T& obj) {
        obj.ParseFromArcadiaStream(input);
    }

    template <class T>
    inline void ReadProtoJson(IInputStream* input, T& obj) {
        using namespace ::google::protobuf::util;

        TString inputString;
        TStringOutput output(inputString);
        TransferData(input, &output);
        output.Finish();

        auto status(JsonStringToMessage(inputString, &obj));
        if (!status.ok()) {
            ythrow TBadRequest() << status.message().ToString();
        }
    }

    template <class T>
    inline void WriteProto(IOutputStream* output, T& obj) {
        obj.SerializeToArcadiaStream(output);
    }

    template <class T>
    inline void WriteProto(TString& buf, const T& obj) {
        TStringOutput output(buf);
        WriteProto(&output, obj);
    }

    template <class T>
    inline void WriteProtoJson(TString& buf, const T& obj) {
        using namespace ::google::protobuf::util;

        auto status(MessageToJsonString(obj, &buf));
        if (!status.ok()) {
            ythrow TBadRequest() << status.message().ToString();
        }
    }

    template <class T>
    inline void WriteProtoJson(IOutputStream* output, const T& obj) {
        TString outputString;
        WriteProtoJson(outputString, obj);
        output->Write(outputString);
    }

    template <class T>
    class TFlatObject: public TPointerBase<TFlatObject<T>, const T>, public TMoveOnly {
    public:
        inline TFlatObject() noexcept
            : Object(nullptr)
        {
        }

        explicit inline TFlatObject(TBuffer&& that) noexcept
            : TFlatObject()
        {
            Buffer = that;
        }

        inline const T* Get() const noexcept {
            return Object;
        }

        const TBuffer& GetBuffer() const noexcept {
            return Buffer;
        }

        inline bool Verify() const noexcept {
           flatbuffers::Verifier verifier(reinterpret_cast<const uint8_t*>(Buffer.Data()), Buffer.Size());
           return Object->Verify(verifier);
        }

        TFlatObject& operator=(TBuffer&& that) noexcept {
            Buffer = that;
            Object = flatbuffers::GetRoot<T>(Buffer.Data());
            return *this;
        }

    private:
        TBuffer Buffer;
        const T* Object;
    };

    template <class T>
    inline void ReadFlatBuffer(IInputStream *input, TFlatObject<T> &ptr) {
        TBuffer output;
        TBufferOutput outputStream(output);
        NBlockCodecs::TDecodedInput decodedStream(input);
        TransferData(&decodedStream, &outputStream);
        outputStream.Finish();
        ptr = std::move(output);
    }

    template <class T>
    inline void ReadFlatBuffer(const TStringBuf &input, TFlatObject<T> &ptr) {
        TMemoryInput inputStream(input);
        ReadFlatBuffer(&inputStream, ptr);
    }

    inline void WriteFlatBuffer(IOutputStream *output, const flatbuffers::FlatBufferBuilder &builder) {
        NBlockCodecs::TCodedOutput compressedOutput(output, CODEC, DEFAULT_BUFFER_LENGTH);
        compressedOutput.Write(ToStringBuf(builder));
        compressedOutput.Finish();
    }

    inline void WriteFlatBuffer(TString &buf, const flatbuffers::FlatBufferBuilder &builder) {
        TStringOutput output(buf);
        WriteFlatBuffer(&output, builder);
    }

    inline void WriteFlatBuffer(NNeh::IRequest *request, const flatbuffers::FlatBufferBuilder &builder) {
        NNeh::TRequestOut output(request);
        WriteFlatBuffer(&output, builder);
    }

    inline TNehRequester::TFuture MakeRequest(
            const TLibrarySettings::TShard& shard, const TString& path,
            const flatbuffers::FlatBufferBuilder& builder,
            TBaseThreadExecutor* pool = nullptr) {
        NNeh::TMessage message;
        message.Addr = TStringBuilder() << "tcp2://" << shard.Host << ":" << shard.Port << path;
        WriteFlatBuffer(message.Data, builder);
        return TNehRequester::Get()->MakeRequest(message, REQUEST_TIMEOUT.ToDeadLine(), pool);
    }
}
