#pragma once

#include <infra/netmon/library/web_server.h>
#include <infra/netmon/library/api_client_helpers.h>
#include <infra/netmon/library/blackbox.h>
#include <infra/netmon/library/metrics.h>

#include <library/cpp/neh/rpc.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/http/cookies/cookies.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json.h>

#include <util/generic/singleton.h>
#include <util/datetime/cputimer.h>
#include <util/string/split.h>
#include <library/cpp/deprecated/atomic/atomic.h>

namespace NNetmon {
    inline TAtomic HttpRequestCount;

    // what content type
    enum class EContentType {
        CodedProtobuf,
        Json
    };

    // code 404 Not Found
    class TNotFoundError: public yexception {
    };

    // code 400 Bad Request
    class TValidationError: public TBadRequest {
    };

    // code 429 Too Many Requests
    class TTooManyRequestsError: public yexception {
    };

    // code 401 Forbidden
    class TUnauthorizedError: public yexception {
    };

    // code 403 Forbidden
    class TForbiddenError: public yexception {
    };

    struct TUserCredentials {
        const TString SessionId;
        const TString OAuthToken;
        const TString UserIp;
        const TString Host;
        const TString UserAgent;
    };

    TUserCredentials ExtractCredentials(const THttpHeaders& headers);
    EContentType ExtractContentType(const THttpHeaders& headers);
    TInstant ExtractStartTime(const THttpHeaders& headers);
    ui64 ExtractReportsCount(const THttpHeaders& headers);

    namespace {
        inline void SafeHttpError(THttpOutput& output, HttpCodes code, const TString& msg) noexcept {
            try {
                output << THttpResponse(code).SetContent(msg);
                output.Finish();
            } catch (...) {
                ERROR_LOG << "Can't send reply to client: " << CurrentExceptionMessage() << Endl;
            }
        }

        template <class T, typename F>
        inline bool WrapHttpError(TIntrusivePtr<T> reply, F&& functor) noexcept {
            const auto& request = *reply->GetClientContext();
            TString exceptionMessage;
            HttpCodes code;
            try {
                functor();
                return true;
            } catch (const TTooManyRequestsError&) {
                exceptionMessage = CurrentExceptionMessage();
                code = HTTP_TOO_MANY_REQUESTS;
            } catch (const TNotFoundError&) {
                exceptionMessage = CurrentExceptionMessage();
                code = HTTP_NOT_FOUND;
            } catch (const TBadRequest&) {
                exceptionMessage = CurrentExceptionMessage();
                code = HTTP_BAD_REQUEST;
            } catch (const NJson::TJsonException&) {
                exceptionMessage = CurrentExceptionMessage();
                code = HTTP_BAD_REQUEST;
            } catch (const TUnauthorizedError&) {
                exceptionMessage = CurrentExceptionMessage();
                code = HTTP_UNAUTHORIZED;
            } catch (const TForbiddenError&) {
                exceptionMessage = CurrentExceptionMessage();
                code = HTTP_FORBIDDEN;
            } catch (...) {
                exceptionMessage = CurrentExceptionMessage();
                code = HTTP_SERVICE_UNAVAILABLE;
                // log errors wisely, bulk errors -> high load on disk
                ERROR_LOG << "HTTP query #" << (size_t)reply.Get() << " failed using"
                          << " " << TypeName<T>() << " at " << reply->GetDuration()
                          << " : " << exceptionMessage << Endl;
            }
            SafeHttpError(request.GetOutput(), code, exceptionMessage);
            return false;
        }

        template <class T>
        inline bool WrapHttpFutureError(TIntrusivePtr<T> reply, const NThreading::TFuture<void>& future) noexcept {
            return WrapHttpError(reply, [&] () {
                future.GetValue();
            });
        }

        template <class T>
        inline void WriteHttpReply(TIntrusivePtr<T> reply) noexcept {
            try {
                reply->WriteResponse(reply->GetClientContext()->GetOutput());
            } catch (...) {
                ERROR_LOG << "Can't write reply for " << TypeName<T>() << ": " << CurrentExceptionMessage() << Endl;
                SafeHttpError(reply->GetClientContext()->GetOutput(), HTTP_SERVICE_UNAVAILABLE, CurrentExceptionMessage());
                return;
            }

            DEBUG_LOG << "HTTP query #" << (size_t)reply.Get() << " finished using"
                      << " " << TypeName<T>() << " at " << reply->GetDuration() << Endl;
        }

        template <class T>
        inline void WriteHttpOptionsReply(TIntrusivePtr<T> reply) noexcept {
            try {
                const auto& input = reply->GetClientContext()->GetInput();
                auto response = THttpResponse();
                response.AddHeader("Access-Control-Allow-Methods", "POST, GET, OPTIONS");

                auto corsRequestHeaders = input.Headers().FindHeader(TStringBuf("Access-Control-Request-Headers"));
                if (corsRequestHeaders) {
                    response.AddHeader("Access-Control-Allow-Headers", corsRequestHeaders->Value());
                }

                auto& output = reply->GetClientContext()->GetOutput();
                output << response;
                output.Finish();
            } catch (...) {
                ERROR_LOG << "Can't write OPTIONS reply for " << TypeName<T>() << ": " << CurrentExceptionMessage() << Endl;
                SafeHttpError(reply->GetClientContext()->GetOutput(), HTTP_SERVICE_UNAVAILABLE, CurrentExceptionMessage());
                return;
            }

            DEBUG_LOG << "HTTP OPTIONS query #" << (size_t)reply.Get() << " finished using"
                      << " " << TypeName<T>() << " at " << reply->GetDuration() << Endl;
        }
    }

    template <class TDispatcher, class TServerContext, class TClientContext>
    class TBaseReply: public TNonCopyable, public TAtomicRefCount<
            TBaseReply<TDispatcher, TServerContext, TClientContext>> {
    public:
        using TServerContextType = TServerContext;
        using TClientContextType = TClientContext;

        TBaseReply(TServerContext& serverContext, const TClientContext& clientContext)
            : ServerContext(serverContext)
            , ClientContext(clientContext)
        {
        }
        virtual ~TBaseReply() = default;

        template <class TReply>
        inline void Dispatch(TIntrusivePtr<TReply> reply) noexcept {
            DEBUG_LOG << "Query #" << (size_t)reply.Get() << " started using"
                      << " " << TypeName<TReply>() << Endl;
            try {
                SingletonWithPriority<TDispatcher, 100006>()->Dispatch(reply);
            } catch(...) {
                ERROR_LOG << "Query using " << TypeName<TReply>() << " failed with: "
                          << CurrentExceptionMessage() << Endl;
            }
        }

        inline const TClientContext& GetClientContext() noexcept {
            return ClientContext;
        }
        inline TServerContext& GetServerContext() noexcept {
            return ServerContext;
        }

        inline TDuration GetDuration() noexcept {
            return Timer.Get();
        }

    private:
        TServerContext& ServerContext;
        const TClientContext ClientContext;
        TSimpleTimer Timer;
    };

    template <class TDispatcher, class TServerContext, class TRequest>
    class TInterconnectReply: public TBaseReply<TDispatcher, TServerContext, NNeh::IRequestRef> {
    public:
        TInterconnectReply(TServerContext& serverContext, const NNeh::IRequestRef& clientContext)
            : TBaseReply<TDispatcher, TServerContext, NNeh::IRequestRef>(serverContext, clientContext)
            , Builder(1024 * 1024)
        {
        }
        virtual ~TInterconnectReply() = default;

        inline TFlatObject<TRequest>& GetRequest() {
            return Request;
        }
        inline const flatbuffers::FlatBufferBuilder& GetBuilder() const {
            return Builder;
        }

    protected:
        TFlatObject<TRequest> Request;
        flatbuffers::FlatBufferBuilder Builder;
    };

    template <class TReply>
    class TInterconnectHandler: public TNonCopyable {
    public:
        TInterconnectHandler(typename TReply::TServerContextType& serverContext)
            : ServerContext(serverContext)
        {
        }

        void ServeRequest(const NNeh::IRequestRef& request) {
            TIntrusivePtr<TReply> reply(MakeIntrusive<TReply>(ServerContext, request));
            reply->Dispatch(reply);
        }

    private:
        typename TReply::TServerContextType& ServerContext;
    };

    class TInterconnectDispatcher: public TNonCopyable {
    public:
        template <class TReply>
        void Dispatch(TIntrusivePtr<TReply> reply) {
            try {
                ReadFlatBuffer(reply->GetClientContext()->Data(), reply->GetRequest());
            } catch (...) {
                reply->GetClientContext()->SendError(
                    NNeh::IRequest::TResponseError::BadRequest,
                    CurrentExceptionMessage());
                return;
            }

            try {
                reply->Process();
                WriteFlatBuffer(reply->GetClientContext().Get(), reply->GetBuilder());
            } catch (...) {
                reply->GetClientContext()->SendError(
                    NNeh::IRequest::TResponseError::ServiceUnavailable,
                    CurrentExceptionMessage());
                ERROR_LOG << "Interconnect query #" << (size_t)reply.Get() << " failed using"
                          << " " << TypeName<TReply>() << " at " << reply->GetDuration()
                          << " : " << CurrentExceptionMessage() << Endl;
                return;
            }

            DEBUG_LOG << "Interconnect query #" << (size_t)reply.Get() << " finished using"
                      << " " << TypeName<TReply>() << " at " << reply->GetDuration() << Endl;
        }
    };

    template <class TDispatcher, class TServerContext>
    using THttpReply = TBaseReply<TDispatcher, TServerContext, TServiceRequest::TRef>;

    template <class TReply>
    class THttpHandler: public IServiceReplier {
    public:
        THttpHandler(typename TReply::TServerContextType& serverContext)
            : ServerContext(serverContext)
        {
        }

        void DoReply(const TServiceRequest::TRef request) {
            AtomicIncrement(HttpRequestCount);

            TIntrusivePtr<TReply> reply(MakeIntrusive<TReply>(ServerContext, request));
            const auto& input = reply->GetClientContext()->GetInput();
            CollectQueueTime(input);
            if (ExtractHttpRequestMethod(input) == TStringBuf("OPTIONS")) {
                // needed to correctly respond to CORS preflight requests
                WriteHttpOptionsReply(reply);
            } else {
                reply->Dispatch(reply);
            }
        }

    private:
        void CollectQueueTime(const THttpInput& input) {
            const auto startTime(ExtractStartTime(input.Headers()));
            const auto now(TInstant::Now());
            if (startTime && startTime < now) {
                TUnistat::Instance().PushSignalUnsafe(
                    ELibrarySignals::HttpServerQueueTime, (now - startTime).MilliSeconds());
            }
        }

        inline TStringBuf ExtractHttpRequestMethod(const THttpInput& input) const {
            return TStringBuf(input.FirstLine()).Before(' ');
        }

        typename TReply::TServerContextType& ServerContext;
    };

    class THttpDispatcher: public TNonCopyable {
    public:
        template <class TReply>
        void Dispatch(TIntrusivePtr<TReply> reply) {
            if (WrapHttpError(reply, [&] () {
                auto& input(reply->GetClientContext()->GetInput());
                reply->PreprocessRequest(input);
                reply->ParseRequest(input);
                reply->Process();
            })) {
                WriteHttpReply(reply);
            }
        }
    };

    class THttpFutureDispatcher: public TNonCopyable {
    public:
        template <class TReply>
        void Dispatch(TIntrusivePtr<TReply> reply) {
            WrapHttpError(reply, [&] () {
                auto& input(reply->GetClientContext()->GetInput());
                reply->PreprocessRequest(input);
                reply->ParseRequest(input);
                reply->Process().Subscribe([reply](const NThreading::TFuture<void>& future) {
                    if (WrapHttpFutureError(reply, future)) {
                        WriteHttpReply(reply);
                    }
                });
            });
        }
    };

    class THttpBlackboxDispatcher: public THttpFutureDispatcher {
    public:
        template <class TReply>
        void Dispatch(TIntrusivePtr<TReply> reply) {
            if (TLibrarySettings::Get()->GetBlackboxUrl().empty()) {
                THttpFutureDispatcher::Dispatch(reply);
                return;
            }

            WrapHttpError(reply, [&] () {
                reply->Authorize(reply->GetClientContext()->GetInput(), Blackbox).Subscribe(
                    [reply, this] (const NThreading::TFuture<void>& future) {
                        if (WrapHttpFutureError(reply, future)) {
                            THttpFutureDispatcher::Dispatch(reply);
                        }
                    }
                );
            });
        }

    private:
        TBlackbox Blackbox;
    };

    template <class TDispatcher, class TServerContext>
    class TJsonHttpReply: public THttpReply<TDispatcher, TServerContext> {
    public:
        TJsonHttpReply(TServerContext& serverContext, const TServiceRequest::TRef& clientContext)
            : THttpReply<TDispatcher, TServerContext>(serverContext, clientContext)
        {
        }

        virtual void PreprocessRequest(THttpInput&) {
        }
        virtual void ParseRequest(THttpInput& input) {
            TBufferOutput output;
            TransferData(&input, &output);

            const TBuffer& buf = output.Buffer();
            if (!buf.Empty()) {
                NJson::ReadJsonFastTree(TStringBuf(buf.Data(), buf.Size()), &Request, true);
            }
        }
        virtual void WriteResponse(THttpOutput& output) {
            output << THttpResponse()
                .SetContentType(TStringBuf("application/json"))
                .SetContent(Response.Str());
            output.Finish();
        }
        inline NJson::TJsonValue& GetRequest() {
            return Request;
        }
        inline NJsonWriter::TBuf& GetResponse() {
            return Response;
        }

    private:
        NJson::TJsonValue Request;
        NJsonWriter::TBuf Response;
    };

    template <class TDispatcher, class TServerContext>
    class TCompressedJsonHttpReply: public TJsonHttpReply<TDispatcher, TServerContext> {
    public:
        using TJsonHttpReply<TDispatcher, TServerContext>::TJsonHttpReply;

        void WriteResponse(THttpOutput& output) override {
            output.EnableCompression(true);
            TJsonHttpReply<TDispatcher, TServerContext>::WriteResponse(output);
        }
    };


    template <class TServerContext>
    class TAuthorizedHttpReply : public TBaseReply<THttpBlackboxDispatcher, TServerContext, TServiceRequest::TRef> {
    public:
        using TBaseReply<THttpBlackboxDispatcher, TServerContext, TServiceRequest::TRef>::TBaseReply;

        inline const TUserState& GetUserState() noexcept {
            return UserState;
        }

        NThreading::TFuture<void> Authorize(THttpInput& input, TBlackbox& blackbox) {
            const auto credentials(ExtractCredentials(input.Headers()));
            auto callback = [this] (const NThreading::TFuture<TUserState>& future) {
                UserState = future.GetValue();
            };
            if (!credentials.OAuthToken.empty()) {
                return blackbox.FindByOAuthToken(credentials.OAuthToken, credentials.UserIp).Apply(callback);
            } else {
                return blackbox.FindBySessionId(credentials.SessionId, credentials.UserIp, credentials.Host).Apply(callback);
            }
        }

    private:
        TUserState UserState;
    };

    template <class TServerContext>
    class TAuthorizedJsonHttpReply: public TAuthorizedHttpReply<TServerContext> {
    public:
        TAuthorizedJsonHttpReply(TServerContext& serverContext, const TServiceRequest::TRef& clientContext)
            : TAuthorizedHttpReply<TServerContext>(serverContext, clientContext)
        {
        }

        virtual void PreprocessRequest(THttpInput&) {
        }
        virtual void ParseRequest(THttpInput& input) {
            TBufferOutput output;
            TransferData(&input, &output);

            const TBuffer& buf = output.Buffer();
            if (!buf.Empty()) {
                NJson::ReadJsonFastTree(TStringBuf(buf.Data(), buf.Size()), &Request, true);
            }
        }
        virtual void WriteResponse(THttpOutput& output) {
            auto response(THttpResponse()
                .SetContentType(TStringBuf("application/json"))
                .SetContent(Response.Str()));
            const auto& userState(this->GetUserState());
            if (userState.Login && userState.YandexUid) {
                response
                    .AddHeader("X-Netmon-Login", userState.Login)
                    .AddHeader("X-Netmon-YandexUid", userState.YandexUid);
            }
            output << response;
            output.Finish();
        }

        inline NJson::TJsonValue& GetRequest() {
            return Request;
        }
        inline NJsonWriter::TBuf& GetResponse() {
            return Response;
        }

    private:
        NJson::TJsonValue Request;
        NJsonWriter::TBuf Response;
    };

    template <class TDispatcher, class TServerContext, class TRequest, class TResponse>
    class TProtoHttpReply: public THttpReply<TDispatcher, TServerContext> {
    public:
        using THttpReply<TDispatcher, TServerContext>::THttpReply;

        virtual void PreprocessRequest(THttpInput&) {
        }
        void ParseRequest(THttpInput& input) {
            ContentType = ExtractContentType(input.Headers());
            switch (ContentType) {
                case EContentType::CodedProtobuf: {
                    return ReadProto(&input, Request);
                }
                case EContentType::Json: {
                    return ReadProtoJson(&input, Request);
                }
            }
        }
        void WriteResponse(THttpOutput& output) {
            THttpResponse response;

            TString content;
            switch (ContentType) {
                case EContentType::CodedProtobuf: {
                    WriteProto(content, Response);
                    response.SetContentType(TStringBuf("application/x-protobuf"));
                    break;
                }
                case EContentType::Json: {
                    WriteProtoJson(content, Response);
                    response.SetContentType(TStringBuf("application/json"));
                    break;
                }
            }

            output << response.SetContent(content);
            output.Finish();
        }
        inline const TRequest& GetRequest() {
            return Request;
        }
        inline TResponse& GetResponse() {
            return Response;
        }

    private:
        EContentType ContentType = EContentType::Json;
        TRequest Request;
        TResponse Response;
    };
}
