#include <infra/netmon/library/blackbox.h>
#include <infra/netmon/library/requester.h>
#include <infra/netmon/library/settings.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/cache/cache.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <util/digest/multi.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <util/string/builder.h>
#include <util/random/easy.h>

namespace NNetmon {
    namespace {
        const TDuration REQUEST_TIMEOUT = TDuration::Seconds(5);
        const TDuration TVM_REQUEST_TIMEOUT = TDuration::Seconds(1);
        const TDuration POSITIVE_TTL = TDuration::Seconds(3600);
        const TDuration ERROR_TTL = TDuration::Seconds(15);
        const TDuration MAX_JITTER = TDuration::Seconds(30);
    }

    struct TCacheKey {
        const TString Token;
        const TString UserIp;
        const TString HostName;

        inline bool operator==(const TCacheKey& rhs) const {
            return Token == rhs.Token && UserIp == rhs.UserIp && HostName == rhs.HostName;
        }
    };

    struct TCacheValue {
        const NThreading::TFuture<TUserState> Value{};
        const TInstant Generated{};
    };

    NThreading::TFuture<TUserState> BlackboxRequest(const TCgiParameters& params,
                                                    const TString& serviceTicket) {
        TVector<TString> headers;
        if (!serviceTicket.Empty()) {
            headers.emplace_back(TStringBuilder() << "X-Ya-Service-Ticket: " << serviceTicket);
        }
        return THttpRequester::Get()->MakeRequest(
            TStringBuilder() << TLibrarySettings::Get()->GetBlackboxUrl() << "/blackbox?" << params.Print(),
            headers,
            NHttp::TFetchOptions().SetTimeout(REQUEST_TIMEOUT)
        ).Apply([] (const THttpRequester::TFuture& future) {
            try {
                future.GetValue();
            } catch (...) {
                ythrow yexception() << "blackbox return an error: " << CurrentExceptionMessage();
            }
            NJson::TJsonValue root;
            NJson::ReadJsonFastTree(future.GetValue()->Data, &root, true);
            if (root.Has("oauth") && root["oauth"].Has("scope") && root["oauth"]["scope"].GetStringSafe() != TStringBuf("netmon:api")) {
                return TUserState("", "", false);
            }
            if (root.Has("login") && root["login"].IsString()
                    && root.Has("uid") && root["uid"].Has("value") && root["uid"]["value"].IsString()) {
                return TUserState(
                    root["login"].GetStringSafe(),
                    root["uid"]["value"].GetStringSafe(),
                    root["status"]["id"].GetIntegerSafe() == 0
                );
            } else {
                return TUserState("", "", false);
            }
        });
    }

    NThreading::TFuture<TUserState> OAuthTokenRequest(const TCacheKey& key,
                                                      const TString& serviceTicket = "") {
        TCgiParameters params;
        params.InsertEscaped("method", TStringBuf("oauth"));
        params.InsertEscaped("oauth_token", key.Token);
        params.InsertEscaped("userip", key.UserIp);
        params.InsertEscaped("format", TStringBuf("json"));
        return BlackboxRequest(params, serviceTicket);
    }

    NThreading::TFuture<TUserState> SessionIdRequest(const TCacheKey& key,
                                                     const TString& serviceTicket = "") {
        TCgiParameters params;
        params.InsertEscaped("method", TStringBuf("sessionid"));
        params.InsertEscaped("sessionid", key.Token);
        params.InsertEscaped("host", key.HostName);
        params.InsertEscaped("userip", key.UserIp);
        params.InsertEscaped("format", TStringBuf("json"));
        return BlackboxRequest(params, serviceTicket);
    }

    class TTvmClientLogger: public NTvmAuth::ILogger {
    public:
        TTvmClientLogger(int level)
            : MaxLevel(level)
        {
        }

        void Log(int level, const TString& msg) override {
            if (level > MaxLevel) {
                return;
            }
            TEMPLATE_LOG(static_cast<ELogPriority>(level)) << msg << Endl;
        }

    private:
        const int MaxLevel;
    };

    class TBlackbox::TImpl {
    public:
        using TReply = NThreading::TFuture<TUserState>;

        inline TImpl()
            : Cache(10000)
        {
            if (!TLibrarySettings::Get()->GetTvmAuthToken().Empty()) {
                const auto* librarySettings = TLibrarySettings::Get();
                NTvmAuth::NTvmTool::TClientSettings tvmSettings(
                        librarySettings->GetTvmNetmonAlias());
                tvmSettings.SetHostname(librarySettings->GetTvmAddress());
                tvmSettings.SetPort(librarySettings->GetTvmPort());
                tvmSettings.SetAuthToken(librarySettings->GetTvmAuthToken());
                tvmSettings.SetSocketTimeout(TVM_REQUEST_TIMEOUT);
                tvmSettings.SetConnectTimeout(TVM_REQUEST_TIMEOUT);

                Tvm = MakeHolder<NTvmAuth::TTvmClient>(
                        tvmSettings, MakeIntrusive<TTvmClientLogger>(TLOG_ERR));
            }
        }

        inline TMaybe<TReply> FindKey(const TCacheKey& key) {
            auto it(Cache.Find(key));
            auto now(TInstant::Now());
            if (it != Cache.End()) {
                auto jitter(TDuration::MicroSeconds(RandomNumber<ui64>(MAX_JITTER.MicroSeconds())));
                if (it->Value.HasException() && it->Generated + ERROR_TTL + jitter > now) {
                    return it->Value;
                } else if (it->Value.HasValue() && it->Generated + POSITIVE_TTL + jitter > now) {
                    return it->Value;
                } else if (!it->Value.HasValue() && !it->Value.HasException()) {
                    return it->Value;
                }
            }
            return {};
        }

        inline NThreading::TFuture<TUserState> FindBySessionId(
                const TString& sessionId, const TString& userIp, const TString& hostName) noexcept {
            if (sessionId.empty() || userIp.empty() || hostName.empty()) {
                return NThreading::MakeFuture(TUserState("", "", false));
            }
            TCacheKey key{sessionId, userIp, hostName};
            auto cached(FindKey(key));
            if (cached.Defined()) {
                return cached.GetRef();
            } else {
                TCacheValue value{SessionIdRequest(key, GetServiceTicket()), TInstant::Now()};
                Cache.Insert(key, value);
                return value.Value;
            }
        }

        inline NThreading::TFuture<TUserState> FindByOAuthToken(
                const TString& oAuthToken, const TString& userIp) noexcept {
            if (oAuthToken.empty() || userIp.empty()) {
                return NThreading::MakeFuture(TUserState("", "", false));
            }
            TCacheKey key{oAuthToken, userIp, ""};
            auto cached(FindKey(key));
            if (cached.Defined()) {
                return cached.GetRef();
            } else {
                TCacheValue value{OAuthTokenRequest(key, GetServiceTicket()), TInstant::Now()};
                Cache.Insert(key, value);
                return value.Value;
            }
        }

    private:
        inline TString GetServiceTicket() {
            if (Tvm) {
                const TString& blackboxAlias = TLibrarySettings::Get()->GetTvmBlackboxAlias();
                return Tvm->GetServiceTicketFor(blackboxAlias);
            } else {
                return "";
            }
        }

        TLRUCache<TCacheKey, TCacheValue> Cache;
        THolder<NTvmAuth::TTvmClient> Tvm;
    };

    TBlackbox::TBlackbox()
        : Impl(MakeHolder<TImpl>())
    {
    }

    TBlackbox::~TBlackbox() {
    }

    NThreading::TFuture<TUserState> TBlackbox::FindBySessionId(
            const TString& sessionId, const TString& userIp, const TString& hostName) noexcept {
        return Impl.Own()->FindBySessionId(sessionId, userIp, hostName);
    }

    NThreading::TFuture<TUserState> TBlackbox::FindByOAuthToken(
            const TString& oAuthToken, const TString& userIp) noexcept {
        return Impl.Own()->FindByOAuthToken(oAuthToken, userIp);
    }
}

template <>
class THash<NNetmon::TCacheKey> {
public:
    inline size_t operator()(const NNetmon::TCacheKey& key) const {
        return MultiHash(key.Token, key.UserIp, key.HostName);
    }
};
