#include <infra/netmon/library/blackbox.h>
#include <infra/netmon/library/api_handler_helpers.h>
#include <infra/netmon/library/settings.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/json/writer/json.h>

#include <util/generic/xrange.h>

using namespace NNetmon;

namespace {
    struct TBlackboxContext : public TNonCopyable {
        TAtomic Served = 0;
    };

    class TTestBlackboxReply: public THttpReply<THttpDispatcher, TBlackboxContext> {
    public:
        TTestBlackboxReply(TBlackboxContext& serverContext, const TServiceRequest::TRef& clientContext)
            : THttpReply<THttpDispatcher, TBlackboxContext>(serverContext, clientContext)
        {
        }

        void PreprocessRequest(THttpInput&) {
        }
        void ParseRequest(THttpInput&) {
        }

        void Process() {
            AtomicIncrement(GetServerContext().Served);
            Response
                .BeginObject()
                .WriteKey("login")
                .WriteString("deadbeaf")
                .WriteKey("status")
                .BeginObject()
                .WriteKey("id")
                .WriteInt(0)
                .EndObject()
                .WriteKey("uid")
                .BeginObject()
                .WriteKey("value")
                .WriteString("123456")
                .EndObject()
                .EndObject();
        }

        void WriteResponse(THttpOutput& output) {
            output << THttpResponse().SetContent(Response.Str());
            output.Finish();
        }

    private:
        NJsonWriter::TBuf Response;
    };
}

class TBlackboxTest: public TTestBase {
    UNIT_TEST_SUITE(TBlackboxTest);
    UNIT_TEST(TestSessionId)
    UNIT_TEST_SUITE_END();

private:
    inline void TestSessionId() {
        TPortManager portManager;
        ui16 port(portManager.GetPort(12345));

        TBlackboxContext context;
        THttpHandler<TTestBlackboxReply> service(context);

        TString host("localhost");
        TWebServer server(THttpServerOptions().SetHost(host).SetPort(port).SetThreads(1));
        server.Add("/blackbox", service);
        TWebServer::TThreadGuard serverGuard(server);

        TLibrarySettings::Get()->SetBlackboxUrl(TStringBuilder() << "http://" << host << ":" << port);

        TBlackbox blackbox;
        for (const auto idx : xrange(10)) {
            auto state(blackbox.FindBySessionId("1", "2", "3").GetValue(TDuration::Max()));
            UNIT_ASSERT_EQUAL(state.Login, "deadbeaf");
            UNIT_ASSERT_EQUAL(state.YandexUid, "123456");
            UNIT_ASSERT(state.IsActive);
            Y_UNUSED(idx);
        }

        UNIT_ASSERT_EQUAL(AtomicGet(context.Served), 1);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TBlackboxTest);
