#pragma once

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/generic/ptr.h>
#include <util/system/spinlock.h>

namespace NNetmon {
    template <class T, class TGuard = TGuard<TAdaptiveLock>>
    class TOwnedBox: public TPointerBase<TOwnedBox<T>, T> {
    public:
        TOwnedBox(T* value, TGuard&& guard)
            : Value(value)
            , Guard(std::forward<TGuard>(guard))
        {
        }

        inline T* Get() const noexcept {
            return Value;
        }

        inline const TGuard& GetGuard() {
            return Guard;
        }

        inline void Release() {
            Guard.Release();
        }

    private:
        T* Value;
        TGuard Guard;

        void operator[](size_t) = delete;
    };

    template <class T>
    class TPlainLockedBox: public TNonCopyable {
    public:
        using TValue = T;

        template <typename... Args>
        TPlainLockedBox(Args&&... args)
            : Value(std::forward<Args>(args)...)
        {
        }

        inline TOwnedBox<T> Own() const {
            TGuard<TAdaptiveLock> guard(Lock);
            return TOwnedBox<T>(&Value, std::move(guard));
        }

    private:
        mutable TValue Value;
        mutable TAdaptiveLock Lock;
    };

    template <class T>
    class TLazyLockedBox: public TNonCopyable {
    public:
        using TValue = T;

        TLazyLockedBox(THolder<T> holder)
            : Value(std::move(holder))
        {
        }

        inline TOwnedBox<T> Own() const {
            TGuard<TAdaptiveLock> guard(Lock);
            return TOwnedBox<T>(Value.Get(), std::move(guard));
        }

    private:
        mutable THolder<TValue> Value;
        mutable TAdaptiveLock Lock;
    };

    template <class T>
    class TIntrusiveLockedBox: public TNonCopyable {
    public:
        using TValue = T;
        using TValueRef = TIntrusivePtr<T>;
        using TConstValueRef = TIntrusiveConstPtr<T>;

        TIntrusiveLockedBox() {
        }

        TIntrusiveLockedBox(TValueRef&& value)
            : Value(std::move(value))
        {
        }

        template <typename... Args>
        TIntrusiveLockedBox(Args&&... args)
            : Value(MakeIntrusive<T>(std::forward<Args>(args)...))
        {
        }

        inline TValueRef Get() {
            TGuard<TAdaptiveLock> guard(Lock);
            return Value;
        }

        inline TConstValueRef Get() const {
            TGuard<TAdaptiveLock> guard(Lock);
            return Value;
        }

        inline TOwnedBox<T> Own() const {
            TGuard<TAdaptiveLock> guard(Lock);
            return TOwnedBox<T>(Value.Get(), std::move(guard));
        }

        inline void Swap(TValueRef& other) {
            TGuard<TAdaptiveLock> guard(Lock);
            Value.Swap(other);
        }

    private:
        TValueRef Value;
        mutable TAdaptiveLock Lock;
    };

    template <class T>
    class TAtomicLockedBox: public TAtomicRefCount<TAtomicLockedBox<T>>, public TNonCopyable {
    public:
        using TRef = TIntrusivePtr<TAtomicLockedBox<T>>;
        using TValue = T;
        using TValueRef = TAtomicSharedPtr<T>;
        using TConstValueRef = TAtomicSharedPtr<const T>;

        template <typename... Args>
        static inline TAtomicLockedBox<T>::TRef Make(Args&&... args) {
            return MakeIntrusive<TAtomicLockedBox<T>>(std::forward<Args>(args)...);
        }

        template <typename... Args>
        TAtomicLockedBox(Args&&... args)
            : Value(MakeAtomicShared<T>(std::forward<Args>(args)...))
        {
        }

        inline TConstValueRef Get() const {
            TGuard<TAdaptiveLock> guard(Lock);
            return Value;
        }

        inline TOwnedBox<T> Own() const {
            TGuard<TAdaptiveLock> guard(Lock);
            return TOwnedBox<T>(Value.Get(), std::move(guard));
        }

        template <class TFunc>
        inline TConstValueRef Apply(TFunc&& func) {
            TGuard<TAdaptiveLock> guard(Lock);
            TConstValueRef oldValue(Value);
            TValueRef newValue(func(oldValue));
            Value.Swap(newValue);
            return Value;
        }

        inline void Swap(TValueRef& other) {
            TGuard<TAdaptiveLock> guard(Lock);
            Value.Swap(other);
        }

    private:
        TValueRef Value;
        mutable TAdaptiveLock Lock;
    };

    template <class T>
    class TRWLockedBox : public TNonCopyable {
    public:
        using TValue = T;
        using TReadOwnedBox = TOwnedBox<T, TLightReadGuard>;
        using TWriteOwnedBox = TOwnedBox<T, TLightWriteGuard>;

        template <typename... Args>
        TRWLockedBox(Args&&... args)
            : Value(std::forward<Args>(args)...)
        {
        }

        inline TReadOwnedBox OwnRead() const {
            TLightReadGuard guard(Lock);
            return {&Value, std::move(guard)};
        }

        inline TWriteOwnedBox OwnWrite() const {
            TLightWriteGuard guard(Lock);
            return {&Value, std::move(guard)};
        }

    private:
        mutable TValue Value;
        mutable TLightRWLock Lock;
    };
}
