#include <infra/netmon/library/clickhouse/client.h>

#include <infra/netmon/library/thread_pool.h>
#include <infra/netmon/library/settings.h>

#include <library/cpp/logger/global/global.h>

#include <util/datetime/cputimer.h>
#include <util/string/subst.h>
#include <util/string/strip.h>
#include <util/random/random.h>

namespace NNetmon {
    namespace {
        TString ToPrettyQuery(const TString& query) {
            TString output(query);
            SubstGlobal(output, '\n', ' ');
            while (SubstGlobal(output, "  ", "")) {
            }
            return StripString(output);
        }
    }

    TClickhouseClient::TQueryOptions& TClickhouseClient::TQueryOptions::RandomShardIndex() noexcept {
        ShardIndex_ = RandomNumber(TLibrarySettings::Get()->GetClickHouseShards().size());
        return *this;
    }

    class TClickhouseClient::TImpl {
    public:
        class TAbstractQuery : public TNonCopyable {
        public:
            using TRef = THolder<TAbstractQuery>;

            static inline void WrappedExecute(TAbstractQuery* ptr) {
                TRef self(ptr);
                TSimpleTimer timer;
                const auto query(ToPrettyQuery(self->Options_.Query()));
                DEBUG_LOG << "Query '" << query << "' sent to ClickHouse" << Endl;
                auto client(self->Impl_->ClientPools.at(self->Options_.ShardIndex())->Take());
                self->Execute(*client);
                DEBUG_LOG << "Reply to query '" << query << "' received for " << timer.Get() << Endl;
            }

            inline TAbstractQuery(const TImpl* impl, const TClickhouseClient::TQueryOptions& options) noexcept
                : Impl_(impl)
                , Options_(options)
            {
            }

            virtual ~TAbstractQuery() {
            }

            virtual void Execute(NClickHouse::TClient& client) = 0;

        private:
            const TImpl* Impl_;

        protected:
            const TClickhouseClient::TQueryOptions Options_;
        };

        class TExecuteQuery : public TAbstractQuery {
        public:
            using TAbstractQuery::TAbstractQuery;

            void Execute(NClickHouse::TClient& client) override {
                NClickHouse::TQuery query(Options_.Query());
                client.Execute(query);
            }
        };

        class TInsertQuery : public TAbstractQuery {
        public:
            inline TInsertQuery(const TImpl* impl,
                                const TClickhouseClient::TQueryOptions& options,
                                NClickHouse::TBlock&& block) noexcept
                : TAbstractQuery(impl, options)
                , Block(std::move(block))
            {
            }

            void Execute(NClickHouse::TClient& client) override {
                client.Insert(Options_.Query(), Block);
            }

        private:
            const NClickHouse::TBlock Block;
        };

        class TSelectQuery : public TAbstractQuery {
        public:
            inline TSelectQuery(const TImpl* impl,
                                const TClickhouseClient::TQueryOptions& options,
                                NClickHouse::TSelectCallback cb) noexcept
                : TAbstractQuery(impl, options)
                , Callback(cb)
            {
            }

            void Execute(NClickHouse::TClient& client) override {
                client.Select(Options_.Query(), Callback);
            }

        private:
            const NClickHouse::TSelectCallback Callback;
        };

        inline TImpl()
            : Executor("ClickhouseClient", TLibrarySettings::Get()->GetClickhousePoolSize())
        {
            const auto& username(TLibrarySettings::Get()->GetClickHouseUsername());
            const auto& password(TLibrarySettings::Get()->GetClickHousePassword());
            for (const auto& shard : TLibrarySettings::Get()->GetClickHouseShards()) {
                NClickHouse::TClientOptions options;
                options
                    .SetHost(shard.Host)
                    .SetPort(shard.Port)
                    .SetConnectTimeout(TDuration::Seconds(1))
                    .SetRequestTimeout(TDuration::Seconds(120))
                    .SetCompressionMethod(
                        TLibrarySettings::Get()->ShouldClickHouseUseCompression()
                            ? NClickHouse::ECompressionMethod::LZ4
                            : NClickHouse::ECompressionMethod::None
                    );
                if (!username.empty()) {
                    options.SetUser(username);
                }
                if (!password.empty()) {
                    options.SetPassword(password);
                }
                ClientPools.emplace_back(TClickhouseClientPool::Make(options));
            }
        }

        inline TFuture Execute(TAbstractQuery::TRef query) noexcept {
            return Executor.Add([ptr = query.Release()]() {
                TAbstractQuery::WrappedExecute(ptr);
            });
        }

    private:
        TBaseThreadExecutor Executor;
        TVector<TClickhouseClientPool::TRef> ClientPools;
    };

    TClickhouseClient::TClickhouseClient()
        : Impl(MakeHolder<TImpl>())
    {
    }

    TClickhouseClient::~TClickhouseClient()
    {
    }

    TClickhouseClient::TFuture TClickhouseClient::Select(const TQueryOptions& options, NClickHouse::TSelectCallback cb) noexcept {
        return Impl->Execute(MakeHolder<TImpl::TSelectQuery>(Impl.Get(), options, cb));
    }

    TClickhouseClient::TFuture TClickhouseClient::Insert(const TQueryOptions& options, NClickHouse::TBlock&& block) noexcept {
        return Impl->Execute(MakeHolder<TImpl::TInsertQuery>(Impl.Get(), options, std::move(block)));
    }

    TClickhouseClient::TFuture TClickhouseClient::Execute(const TQueryOptions& options) noexcept {
        return Impl->Execute(MakeHolder<TImpl::TExecuteQuery>(Impl.Get(), options));
    }
}
