#include <infra/netmon/library/clickhouse/tables.h>

#include <library/cpp/logger/global/global.h>

#include <util/generic/xrange.h>
#include <util/string/builder.h>
#include <util/string/subst.h>

namespace NNetmon {
    TTableFactory::TTableFactory(std::size_t shardIndex)
        : ShardIndex(shardIndex)
        , Promise(NThreading::NewPromise<void>())
    {
    }

    TClickhouseClient::TFuture TTableFactory::Run() {
        TClickhouseClient::TQueryOptions options("SHOW TABLES");
        auto future(TClickhouseClient::Get()->Select(options.SetShardIndex(ShardIndex), [this] (const NClickHouse::TBlock& block) {
            const auto rows(TranslateClickhouseBlock<std::tuple<TString>>(block));
            for (const auto& row : rows) {
                ExitingTables.insert(std::get<0>(row));
            }
        }));
        future.Subscribe([this](const TClickhouseClient::TFuture& future) {
            try {
                future.GetValue(TDuration::Max());
            } catch (...) {
                Promise.SetValue();
                ERROR_LOG << "Unable to get tables from ClickHouse: "
                          << CurrentExceptionMessage()
                          << Endl;
                return;
            }

            Execute();

            WaitExceptionOrAll(Futures).Subscribe([this](const TClickhouseClient::TFuture&) {
                Promise.SetValue();
            });
        });
        return Promise.GetFuture();
    }

    TTableMaintainer::TTableMaintainer(
            std::size_t shardIdx, const TStringBuf& tableTemplate,
            const TStringBuf& tablePrefix, const char* tableFormat,
            std::size_t tableCount)
        : TTableFactory(shardIdx)
        , TableTemplate(tableTemplate)
        , TablePrefix(tablePrefix)
        , TableFormat(tableFormat)
        , TableCount(tableCount)
    {
    }

    void TTableMaintainer::Execute() {
        for (auto it(ExitingTables.begin()); it != ExitingTables.end();) {
            if (it->StartsWith(TablePrefix)) {
                ++it;
            } else {
                ExitingTables.erase(it++);
            }
        }

        DropTables();
        CreateTables();
    }

    void TTableMaintainer::DropTables() {
        while (ExitingTables.size() > TableCount) {
            auto it = ExitingTables.begin();
            Futures.emplace_back(TClickhouseClient::Get()->Execute(
                TClickhouseClient::TQueryOptions(TStringBuilder() << "DROP TABLE " << *it).SetShardIndex(ShardIndex)
            ).Apply([](const TClickhouseClient::TFuture& future_) {
                try {
                    future_.GetValue();
                } catch (...) {
                    ERROR_LOG << "Unable to delete table from ClickHouse: "
                              << CurrentExceptionMessage()
                              << Endl;
                }
            }));
            ExitingTables.erase(it);
        }
    }

    void TTableMaintainer::CreateTables() {
        TSet<TString> tables;

        for (const auto& delta : xrange(2)) {
            struct tm generatedTs;
            (TInstant::Now() + TDuration::Hours(delta)).LocalTime(&generatedTs);
            tables.emplace(TStringBuilder() << TablePrefix << Strftime(TableFormat, &generatedTs));
        }

        for (const auto& tableName : tables) {
            MakeTable(tableName);
        }
    }

    void TTableMaintainer::MakeTable(const TString& tableName) {
        if (ExitingTables.find(tableName) == ExitingTables.end()) {
            TString query(TableTemplate);
            SubstGlobal(query, "{tableName}", tableName);
            Futures.emplace_back(TClickhouseClient::Get()->Execute(
                TClickhouseClient::TQueryOptions(query).SetShardIndex(ShardIndex)
            ).Apply([](const TClickhouseClient::TFuture& future_) {
                try {
                    future_.GetValue();
                } catch (...) {
                    ERROR_LOG << "Unable to create table from ClickHouse: "
                              << CurrentExceptionMessage()
                              << Endl;
                }
            }));
        }
    }

    TTableCreator::TTableCreator(std::size_t shardIdx, const TTableMap& tableMap)
        : TTableFactory(shardIdx)
        , TableMap(tableMap)
    {
    }

    void TTableCreator::Execute() {
        for (const auto& pair : TableMap) {
            if (ExitingTables.find(pair.first) == ExitingTables.end()) {
                Futures.emplace_back(TClickhouseClient::Get()->Execute(
                    TClickhouseClient::TQueryOptions(pair.second).SetShardIndex(ShardIndex)
                ).Apply([](const TClickhouseClient::TFuture& future_) {
                    try {
                        future_.GetValue();
                    } catch (...) {
                        ERROR_LOG << "Unable to create table from ClickHouse: "
                                  << CurrentExceptionMessage()
                                  << Endl;
                    }
                }));
            }
        }
    }
}
