#pragma once

#include <library/cpp/clickhouse/client/block.h>
#include <library/cpp/clickhouse/client/columns/numeric.h>
#include <library/cpp/clickhouse/client/columns/date.h>
#include <library/cpp/clickhouse/client/columns/string.h>

#include <util/generic/xrange.h>
#include <util/generic/yexception.h>
#include <util/system/type_name.h>
#include <util/datetime/base.h>

namespace NNetmon {
    namespace {
        template <typename TRow>
        class TTranslator {
        public:
            inline TTranslator(const NClickHouse::TBlock& block, TVector<TRow>& result)
                : Block(block)
                , Result(result)
            {
                constexpr std::size_t N = std::tuple_size<TRow>::value;
                if (Block.GetColumnCount() != N) {
                    ythrow yexception() << "wrong column count, given "
                                        << Block.GetColumnCount() << ", expected " << N;
                }
                ForEachColumnInBlock(std::make_index_sequence<N>{});
            }

        private:
            template <std::size_t I,
                      typename TElement,
                      typename TExpectedElement,
                      typename TColumn,
                      std::enable_if_t<!std::is_same<TElement, TExpectedElement>::value, int> = 0>
            inline void CastToResult() {
                ythrow yexception() << "unsupported column type";
            }

            template <std::size_t I,
                      typename TElement,
                      typename TExpectedElement,
                      typename TColumn,
                      std::enable_if_t<std::is_same<TElement, TExpectedElement>::value, int> = 0>
            inline void CastToResult() {
                NClickHouse::TColumnRef currentColumn(Block[I]);
                auto column(currentColumn->As<TColumn>());
                for (auto index : xrange(Block.GetRowCount())) {
                    std::get<I>(Result[index]) = column->At(index);
                }
            }

            template <std::size_t I, typename TElement>
            inline void ColumnToResult() {
                using ECode = NClickHouse::TType::ECode;
                switch (Block[I]->Type()->GetCode()) {
                    case ECode::Int8: {
                        return CastToResult<I, TElement, i8, NClickHouse::TColumnInt8>();
                    }
                    case ECode::Int16: {
                        return CastToResult<I, TElement, i16, NClickHouse::TColumnInt16>();
                    }
                    case ECode::Int32: {
                        return CastToResult<I, TElement, i32, NClickHouse::TColumnInt32>();
                    }
                    case ECode::Int64: {
                        return CastToResult<I, TElement, i64, NClickHouse::TColumnInt64>();
                    }
                    case ECode::UInt8: {
                        return CastToResult<I, TElement, ui8, NClickHouse::TColumnUInt8>();
                    }
                    case ECode::UInt16: {
                        return CastToResult<I, TElement, ui16, NClickHouse::TColumnUInt16>();
                    }
                    case ECode::UInt32: {
                        return CastToResult<I, TElement, ui32, NClickHouse::TColumnUInt32>();
                    }
                    case ECode::UInt64: {
                        return CastToResult<I, TElement, ui64, NClickHouse::TColumnUInt64>();
                    }
                    case ECode::Float32: {
                        return CastToResult<I, TElement, float, NClickHouse::TColumnFloat32>();
                    }
                    case ECode::Float64: {
                        return CastToResult<I, TElement, double, NClickHouse::TColumnFloat64>();
                    }
                    case ECode::Date: {
                        return CastToResult<I, TElement, std::time_t, NClickHouse::TColumnDate>();
                    }
                    case ECode::DateTime: {
                        return CastToResult<I, TElement, std::time_t, NClickHouse::TColumnDateTime>();
                    }
                    case ECode::String: {
                        return CastToResult<I, TElement, TString, NClickHouse::TColumnString>();
                    }
                    case ECode::FixedString: {
                        return CastToResult<I, TElement, TString, NClickHouse::TColumnFixedString>();
                    }
                    default: {
                        ythrow yexception() << "unsupported column type";
                    }
                }
            }

            template <std::size_t ...I>
            inline void ForEachColumnInBlock(std::index_sequence<I...>) {
                using TSwallow = int[];
                (void)TSwallow{1,
                    (ColumnToResult<I, std::tuple_element_t<I, TRow>>(), int{})...
                };
            }

            const NClickHouse::TBlock& Block;
            TVector<TRow>& Result;
        };
    }

    template <typename T>
    inline TVector<T> TranslateClickhouseBlock(const NClickHouse::TBlock& block) {
        TVector<T> result(block.GetRowCount());
        if (block.GetRowCount() && block.GetColumnCount()) {
            TTranslator<T> translator(block, result);
        }
        return result;
    }
}

