#pragma once

#include <infra/netmon/library/web_server.h>
#include <infra/netmon/library/scheduler.h>
#include <infra/netmon/library/requester.h>
#include <infra/netmon/library/thread_pool.h>
#include <infra/netmon/library/metrics.h>
#include <infra/netmon/library/settings.h>

#include <library/cpp/unistat/unistat.h>
#include <library/cpp/neh/neh.h>
#include <library/cpp/neh/multi.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/config/config.h>

#include <util/string/builder.h>
#include <util/system/mutex.h>
#include <util/system/file.h>
#include <util/stream/file.h>

namespace NNetmon {
    class TDaemonHelper;

    class TDaemon: public TNonCopyable {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        TDaemon();
        ~TDaemon();

        void SetHelper(TDaemonHelper* helper);

        void Stop(int signal);
        void ReopenLog(int signal);

        static TDaemon* Get() {
            return SingletonWithPriority<TDaemon, 100005>();
        }

    private:
        static void StopServerStatic(int sig) noexcept {
            TDaemon::Get()->Stop(sig);
        }

        static void ReopenLogStatic(int sig) noexcept {
            TDaemon::Get()->ReopenLog(sig);
        }

        THolder<TDaemonHelper> Helper;
    };

    class TDaemonHelper: public TNonCopyable {
    public:
        TDaemonHelper()
            : NehLoopRef(NNeh::CreateLoop())
            , HttpServer(THttpServerOptions()
                .SetPort(TLibrarySettings::Get()->GetHttpPort())
                .SetThreads(TLibrarySettings::Get()->GetHttpThreadPoolSize())
                .SetMaxQueueSize(TLibrarySettings::Get()->GetHttpMaxQueueSize())
                .SetMaxConnections(TLibrarySettings::Get()->GetHttpMaxConnections())
                .SetListenBacklog(TLibrarySettings::Get()->GetHttpListenBacklog())
                .SetClientTimeout(TLibrarySettings::Get()->GetHttpClientTimeout())
            )
        {
        }

        template <class T>
        void RegisterInterconnectService(const TString& path, T& service) {
            NehLoopRef->Add(
                TStringBuilder() << "tcp2://*:" << TLibrarySettings::Get()->GetInterconnectPort() << path,
                service);
        }

        template <class T>
        void RegisterHttpService(const TString& path, T& service) {
            HttpServer.Add(path, service);
        }

        void Run() {
            TWebServer::TThreadGuard httpServerGuard(HttpServer);
            NehLoopRef->ForkLoop(TLibrarySettings::Get()->GetInterconnectThreadPoolSize());

            Event.Wait();

            NehLoopRef->SyncStopFork();

            TScheduler::Get()->StopTasks();
        }

        void Stop(int signal) {
            TGuard<TMutex> guard(Mutex);
            if (Stopped) {
                return;
            }
            Stopped = true;
            NOTICE_LOG << "Stopping server because catch signal " << signal << Endl;
            Event.Signal();
        }

        void ReopenLog(int signal) {
            if (!GlobalLogInitialized()) {
                return;
            }
            TLoggerOperator<TGlobalLog>::Log().ReopenLog();
            NOTICE_LOG << "Logs was reopened because catch signal " << signal << Endl;
        }

        template <class TProvisioner, class TApplication>
        static int RunFromMain(int argc, const char** argv) {
            using namespace NLastGetopt;

            auto& tass(TUnistat::Instance());
            TLibraryStatsInitializer().Init(tass);

            TProvisioner provisioner;
            provisioner.InitMetrics(tass);

            TOpts opts = NLastGetopt::TOpts::Default();

#ifdef NOC_SLA_BUILD
            opts.SetTitle("Netmon NOC SLA build");
#else
            opts.SetTitle("Netmon production build");
#endif

            TOpt& optHttpPort = opts.AddLongOption('p', "http-port", "http server port to listen.");
            optHttpPort.Optional().RequiredArgument().DefaultValue("14242");

            TOpt& optInterconnectPort = opts.AddLongOption('t', "interconnect-port", "interconnect port to listen.");
            optInterconnectPort.Optional().RequiredArgument().DefaultValue("14243");

            TOpt& optConfigFile = opts.AddLongOption('c', "config", "Path to config file.");
            optConfigFile.Optional().RequiredArgument().DefaultValue("");

            TOpt& optLogFile = opts.AddLongOption("log-file", "Path to log file.");
            optLogFile.Optional().RequiredArgument().DefaultValue("console");

            TOpt& optVerbose = opts.AddLongOption("verbose", "Enable verbose mode.");
            optVerbose.Optional().HasArg(NO_ARGUMENT);

            TOpt& optRequestLogs = opts.AddLongOption("request-log", "Log client and his request.");
            optRequestLogs.Optional().HasArg(NO_ARGUMENT);

            provisioner.ConfigureCommandLine(opts);

            opts.AddHelpOption('h');
            opts.AddVersionOption('v');

            THolder<TOptsParseResult> r;
            try {
                r.Reset(new TOptsParseResult(&opts, argc, argv));
            } catch (...) {
                return 1;
            }

            try {
                DoInitGlobalLog(r->Get(&optLogFile), r->Has(&optVerbose) ? TLOG_DEBUG : TLOG_INFO, false, false);
            } catch (...) {
                Cerr << "Can't initialize logger: " << CurrentExceptionMessage() << Endl;
                return 1;
            }

            if (r->Has(&optConfigFile)) {
                try {
                    const TFile configFile(r->Get(&optConfigFile), EOpenModeFlag::OpenExisting | EOpenModeFlag::RdOnly | EOpenModeFlag::Seq);
                    TFileInput configFileBuffer(configFile);
                    NConfig::TGlobals globals;
                    NConfig::TConfig config = NConfig::TConfig::FromJson(configFileBuffer, globals);
                    TLibrarySettings::Get()->Load(config);
                    provisioner.LoadConfig(config);
                } catch (...) {
                    Cerr << "Can't parse config: " << CurrentExceptionMessage() << Endl;
                    return 1;
                }
            }

            try {
                TLibrarySettings::Get()->SetHttpPort(FromString<ui16>(r->Get(&optHttpPort)));
                TLibrarySettings::Get()->SetInterconnectPort(FromString<ui16>(r->Get(&optInterconnectPort)));
                if (r->Has(&optRequestLogs)) {
                    TLibrarySettings::Get()->EnableRequestLogs();
                }
                provisioner.ParseCommandLine(opts, *r);
            } catch (...) {
                Cerr << "Can't apply config: " << CurrentExceptionMessage() << Endl;
                return 1;
            }

            int exitCode = 0;
            try {
                THolder<TDaemonHelper> daemonHelper(new TDaemonHelper());
                TDaemonHelper* ptr = daemonHelper.Release();
                TDaemon::Get()->SetHelper(ptr);

                INFO_LOG << "Starting " << TypeName<TApplication>() << "..." << Endl;
                TApplication application(*ptr);
                application.Run();
            } catch (...) {
                FATAL_LOG << "Can't run daemon: " << CurrentExceptionMessage() << Endl;
                exitCode = 1;
            }

            TScheduler::Get()->Stop();
            THttpRequester::Get()->Destroy();
            TNehRequester::Get()->Destroy();
            TThreadPool::Get()->Stop();

            return exitCode;
        }

    private:
        bool Stopped = false;
        TMutex Mutex;
        TAutoEvent Event;

        NNeh::IServicesRef NehLoopRef;

        TWebServer HttpServer;
    };

    TDaemon::TDaemon()
        : Helper(nullptr)
    {
        signal(SIGINT, StopServerStatic);
        signal(SIGTERM, StopServerStatic);
        signal(SIGHUP, ReopenLogStatic);
    }

    TDaemon::~TDaemon() {
        INFO_LOG << "Destroying whole application..." << Endl;
        Helper.Reset();
    }

    void TDaemon::SetHelper(TDaemonHelper* helper) {
        Helper.Reset(helper);
    }

    void TDaemon::Stop(int sig) {
        if (Helper) {
            Helper->Stop(sig);
        }
    }

    void TDaemon::ReopenLog(int sig) {
        if (Helper) {
            Helper->ReopenLog(sig);
        }
    }
}
