#pragma once

#include <infra/netmon/library/boxes.h>

#include <util/generic/hash.h>

namespace NNetmon {
    template <typename... Args>
    class TEventHub : public TNonCopyable, public TAtomicRefCount<TEventHub<Args...>> {
    public:
        using TRef = TIntrusivePtr<TEventHub<Args...>>;

        static inline TRef Make() {
            return new TEventHub<Args...>();
        }

    private:
        inline TEventHub()
        {
        }

        inline void Remove(size_t SubscriptionId) const {
            Subscriptions.Own()->Get().erase(SubscriptionId);
        }

    public:
        class TSubscriptionGuard : public TNonCopyable {
        public:
            inline TSubscriptionGuard(const TEventHub<Args...>::TRef hub, size_t subscriptionId)
                : Hub(std::move(hub))
                , SubscriptionId(subscriptionId)
            {
            }

            inline TSubscriptionGuard(TSubscriptionGuard&& other) noexcept
                : Hub(std::move(other.Hub))
                , SubscriptionId(other.SubscriptionId)
            {
            }

            ~TSubscriptionGuard() {
                if (Hub) {
                    Hub->Remove(SubscriptionId);
                }
            }

        private:
            TEventHub<Args...>::TRef Hub;
            const size_t SubscriptionId;
        };

        template <typename F>
        [[nodiscard]] inline TSubscriptionGuard Subscribe(F&& callback) const {
            auto subscriptions(Subscriptions.Own());
            const auto subscriptionId(subscriptions->NewId());
            subscriptions->Get().emplace(subscriptionId, std::move(callback));
            return {const_cast<TEventHub<Args...>*>(this), subscriptionId};
        }

        inline void Notify(const Args&... args) const {
            TCallbacks callbacks(Subscriptions.Own()->Get());
            for (const auto& pair : callbacks) {
                pair.second(args...);
            }
        }

    private:
        using TCallbacks = THashMap<size_t, std::function<void(const Args&...)>>;

        class TContainer : public TNonCopyable {
        public:
            inline const TCallbacks& Get() const noexcept {
                return Callbacks;
            };
            inline TCallbacks& Get() noexcept {
                return Callbacks;
            };

            inline size_t NewId() noexcept {
                return ++LastId;
            }

        private:
            TCallbacks Callbacks;
            size_t LastId = 0;
        };

        TPlainLockedBox<TContainer> Subscriptions;
    };

    using TVoidEventHub = TEventHub<>;
}
