#pragma once

#include <library/cpp/threading/future/future.h>
#include <util/generic/list.h>

#include <util/system/condvar.h>

namespace NNetmon {
    class TOwningFence: public TNonCopyable {
    public:
        TOwningFence() : HolderCount(0) {}

        ~TOwningFence() {
            Wait();
        }

        void Acquire() noexcept {
            auto result = AtomicIncrement(HolderCount);
            Y_VERIFY(result > 0);
        }

        void Release() noexcept {
            auto result = AtomicDecrement(HolderCount);
            Y_VERIFY(result >= 0);
            if (result == 0) {
                TGuard<TMutex> guard(Mutex);
                CondVar.BroadCast();
            }
        }

        inline void Wait() noexcept {
            while (AtomicGet(HolderCount) > 0) {
                TGuard<TMutex> guard(Mutex);
                CondVar.WaitT(Mutex, TDuration::MilliSeconds(10));
            }
        }

    private:
        TAtomic HolderCount;
        TMutex Mutex;
        TCondVar CondVar;
    };

    class TOwningGuard : public TGuard<TOwningFence>, public TAtomicRefCount<TOwningGuard> {
    public:
        TOwningGuard(const TOwningFence &fence) : TGuard<TOwningFence>(fence) {}
    };

    class TFutureFence: public TNonCopyable {
    public:
        template <typename T>
        TFutureFence& Retain(const NThreading::TFuture<T>& future) {
            //No outer locking, because we don't care who will set futures
            AtomicIncrement(_Counter);

            future.Subscribe([this](const NThreading::TFuture<T>&) noexcept {
                if (AtomicDecrement(_Counter) == 0) {
                    TGuard<TAdaptiveLock> guard(Lock);
                    for (auto &p: Promises) {
                        p.SetValue();
                    }
                    Promises.clear();
                }
            });

            return *this;
        }

        NThreading::TFuture<void> Wait() {
            auto promise = NThreading::NewPromise<void>();
            TGuard<TAdaptiveLock> guard(Lock);

            if (AtomicGet(_Counter) == 0lu) {
                promise.SetValue();
            } else {
                Promises.push_back(promise);
            }

            return promise;
        }

    private:
        TList<NThreading::TPromise<void> > Promises;
        TAdaptiveLock Lock;
        TAtomic _Counter = 0;
    };
}
