#include <infra/netmon/library/helpers.h>

#include <library/cpp/regex/pire/regexp.h>
#include <library/cpp/regex/libregex/regexstr.h>

#include <util/generic/algorithm.h>
#include <util/generic/xrange.h>
#include <util/generic/utility.h>
#include <util/system/spinlock.h>
#include <util/network/init.h>

#include <array>
#include <cmath>

namespace NNetmon {
    namespace {
        using TAgeWeights = std::array<double, 600>;

        TAgeWeights ComputeAgeWeights() {
            TAgeWeights weights{};
            for (const auto idx : xrange(weights.max_size())) {
                if (idx) {
                    weights[idx] = 1.0 - Min(Max(std::log10(idx), 0.0), 0.9);
                } else {
                    weights[idx] = 1.0;
                }
            }
            return weights;
        }

        const TAgeWeights AgeWeights = ComputeAgeWeights();

        template <class T>
        static void* ThreadProcWrapper(void* param) {
            return reinterpret_cast<T*>(param)->ThreadProc();
        }
    }

    TInstant RoundInstant(const TInstant& instant, const TDuration& interval) {
        auto ts(instant.Seconds());
        return TInstant::Seconds(ts - ts % interval.Seconds());
    }

    INamedThread::INamedThread(const TString& name, size_t stackSize)
        : TThread(TParams(ThreadProcWrapper<INamedThread>, reinterpret_cast<void*>(this), stackSize).SetName(name))
    {
    }

    TThreadedLoop::TThreadedLoop(const TString& name)
        : TThreadedLoop(name, TDuration::Seconds(1))
    {
    }

    TThreadedLoop::TThreadedLoop(const TString& name, const TDuration& iterationTime)
        : INamedThread(name)
        , ShouldStop(false)
        , IterationTime(iterationTime)
    {
    }

    void* TThreadedLoop::ThreadProc() noexcept {
        while (!ShouldStop) {
            auto started = TInstant::Now();
            try {
                OnLoop();
            } catch (...) {
                ERROR_LOG << "Unhandled exception in loop: " << CurrentExceptionMessage() << Endl;
            }
            auto finished = TInstant::Now();
            auto duration = finished - started;
            if (duration < IterationTime) {
                Event.WaitT(IterationTime - duration);
            }
        }
        return this;
    }

    void TThreadedLoop::ScheduleStop() {
        if (!ShouldStop) {
            ShouldStop = true;
            Event.Signal();
        }
    }

    void TThreadedLoop::Stop() {
        ScheduleStop();
        Join();
    }

    double GetAgeWeight(ui64 age) {
        if (age >= AgeWeights.size()) {
            return 0.1;
        } else {
            return AgeWeights[age];
        }
    }

    bool ValidateGroup(const TString& group) {
        using namespace NRegExp;
        static const TFsm fsm("[\\w\\d_\\-@\\s\\.]+", TFsm::TOptions().SetCaseInsensitive(true));
        return TMatcher(fsm).Match(group).Final();
    }

    bool ValidateId(const TString& id) {
        using namespace NRegExp;
        static const TFsm fsm("[\\w\\d_\\-]+", TFsm::TOptions().SetCaseInsensitive(true));
        return TMatcher(fsm).Match(id).Final();
    }

    TAgentVersion ParseAgentVersion(const TString& version) {
        // something like v0.4.0+2857418
        TRegexStringParser parser("^v([\\d]+)[.]([\\d]+)[.]([\\d]+)[+]([\\d]+)$");

        TAgentVersion result;

        auto callback = [&result](unsigned int pos, const char *s, size_t len) noexcept {
            TStringBuf buf(s, len);
            switch(pos) {
                case 1:
                    std::get<0>(result) = FromString<int>(buf);
                    break;
                case 2:
                    std::get<1>(result) = FromString<int>(buf);
                    break;
                case 3:
                    std::get<2>(result) = FromString<int>(buf);
                    break;
                case 4:
                    std::get<3>(result) = FromString<long>(buf);
                    break;
                default:
                    Y_ASSERT(false);
                    break;
            };
        };

        if (parser.Match(callback, version.data(), version.size())) {
            return result;
        } else {
            return {0, 0, 0, 0};
        }
    }

    namespace {
        union TIpStorage {
            using TUnsignedInt128 = std::pair<ui64, ui64>;

            inline TIpStorage() noexcept {
                std::memset(this, 0, sizeof(TIpStorage));
            }
            inline TIpStorage(const NCommon::TIpAddress& address) noexcept {
                static_assert(sizeof(TUnsignedInt128) == sizeof(TIpStorage), "check struct sizes");
                High() = address.High();
                Low() = address.Low();
            }

            inline TUnsignedInt128& AsUint128() noexcept {
                return *reinterpret_cast<TUnsignedInt128*>(this);
            }
            inline ui64& Low() noexcept {
                return AsUint128().first;
            }
            inline ui64& High() noexcept {
                return AsUint128().second;
            }

            inline void* Addr(ui16 family) noexcept {
                switch (family) {
                    case AF_INET: {
                        return reinterpret_cast<void*>(&In4Addr.s_addr);
                    }
                    case AF_INET6: {
                        return reinterpret_cast<void*>(&In6Addr.s6_addr);
                    }
                    default: {
                        return nullptr;
                    }
                }
            }

            inline TString ToBytes(ui16 family) noexcept {
                const void* addr(Addr(family));
                if (addr) {
                    return {reinterpret_cast<const char*>(addr), Size(family)};
                } else {
                    return {};
                }
            }

            inline TString ToString(ui16 family) noexcept {
                char buf[INET6_ADDRSTRLEN + 1];
                const void* addr(Addr(family));
                if (addr && inet_ntop(family, addr, buf, sizeof(buf))) {
                    return buf;
                } else {
                    return {};
                }
            }

            static inline std::size_t Size(ui16 family) noexcept {
                switch (family) {
                    case AF_INET: {
                        return sizeof(in_addr);
                    }
                    case AF_INET6: {
                        return sizeof(in6_addr);
                    }
                    default: {
                        return 0;
                    }
                }
            }

            static inline NCommon::TIpAddress FromString(ui16 family, const TString& buf) noexcept {
                TIpStorage storage;
                switch (family) {
                    case AF_INET: {
                        inet_pton(AF_INET, buf.c_str(), &storage.In4Addr);
                        break;
                    }
                    case AF_INET6: {
                        inet_pton(AF_INET6, buf.c_str(), &storage.In6Addr);
                        break;
                    }
                    default: {
                        break;
                    }
                }
                return {family, storage.High(), storage.Low()};
            }

            static inline NCommon::TIpAddress FromBytes(const TStringBuf& buf) noexcept {
                TIpStorage storage;
                ui16 family = 0;
                switch (buf.size()) {
                    case sizeof(in_addr): {
                        family = AF_INET;
                        break;
                    }
                    case sizeof(in6_addr): {
                        family = AF_INET6;
                        break;
                    }
                    default: {
                        return {0, 0, 0};
                    }
                }
                std::memcpy(storage.Addr(family), buf.data(), buf.size());
                return {family, storage.High(), storage.Low()};
            }

            in6_addr In6Addr;
            in_addr In4Addr;
        };
    }

    TIpAddress::TIpAddress(ui16 family, const TString& buf) noexcept
        : Address{TIpStorage::FromString(family, buf)}
    {
    }

    TIpAddress::TIpAddress(const TStringBuf& buf) noexcept
        : Address(TIpStorage::FromBytes(buf))
    {
    }

    TIpAddress::TIpAddress(const NCommon::TIpAddress& address) noexcept
        : Address(address)
    {
    }

    TIpAddress::TIpAddress(const TIpAddress& other) noexcept
        : Address(other.Address)
    {
    }

    TString TIpAddress::ToBytes() const noexcept {
        TIpStorage storage(Address);
        return storage.ToBytes(Address.Family());
    }

    TString TIpAddress::ToString() const noexcept {
        TIpStorage storage(Address);
        return storage.ToString(Address.Family());
    }
}
