#pragma once

#include <infra/netmon/idl/helpers.fbs.h>

#include <library/cpp/logger/global/global.h>

#include <util/system/thread.h>
#include <util/system/event.h>

namespace NNetmon {
    TInstant RoundInstant(const TInstant& instant, const TDuration& interval);

    // Based on ISimpleThread from util/system/thread.h
    class INamedThread: public TThread {
    public:
        INamedThread(const TString& name, size_t stackSize = 0);

        virtual ~INamedThread() = default;

        virtual void* ThreadProc() = 0;
    };

    template <class T>
    struct TStartStopOps {
    public:
        static inline void Acquire(T* t) noexcept {
            t->Start();
        }

        static inline void Release(T* t) noexcept {
            t->Stop();
        }
    };

    template <class T>
    using TThreadGuard = TGuard<T, TStartStopOps<T>>;

    class TThreadedLoop: public INamedThread {
    public:
        using TThreadGuard = TThreadGuard<TThreadedLoop>;

        TThreadedLoop(const TString& name);
        TThreadedLoop(const TString& name, const TDuration& iterationTime);
        virtual ~TThreadedLoop() {
        }

        void* ThreadProc() noexcept override;
        void ScheduleStop();
        virtual void Stop();

        bool IsRunning() const {
            return !ShouldStop;
        }

        virtual void OnLoop() = 0;

    private:
        volatile bool ShouldStop;
        TAutoEvent Event;
        TDuration IterationTime;
    };

    struct TCompareUsingLess {
        template <class T>
        static inline bool Compare(const T& l, const T& r) {
            return l < r;
        }
    };

    struct TCompareUsingDeadline {
        template <class T>
        static inline bool Compare(const T& lhs, const T& rhs) {
            return (
                lhs.GetDeadline() < rhs.GetDeadline()
                || (
                    lhs.GetDeadline() == rhs.GetDeadline()
                    && &lhs < &rhs
                )
            );
        }
    };

    double GetAgeWeight(ui64 age);

    bool ValidateGroup(const TString& group);
    bool ValidateId(const TString& id);

    class TAgentVersion : public std::tuple<int, int, int, long> {
    public:
        using tuple::tuple;

        inline int Major() const noexcept {
            return std::get<0>(*this);
        }
        inline int Minor() const noexcept {
            return std::get<1>(*this);
        }
        inline int Patch() const noexcept {
            return std::get<2>(*this);
        }
        inline long Revision() const noexcept {
            return std::get<3>(*this);
        }
    };

    TAgentVersion ParseAgentVersion(const TString& version);

    class TIpAddress {
    public:
        inline TIpAddress() noexcept
            : TIpAddress(0)
        {
        }

        // from string
        explicit TIpAddress(ui16 family, const TString& buf) noexcept;
        // from bytes
        explicit TIpAddress(const TStringBuf& buf) noexcept;
        // from proto
        explicit TIpAddress(const NCommon::TIpAddress& address) noexcept;
        // copy constructor
        explicit TIpAddress(const TIpAddress& address) noexcept;

        TString ToBytes() const noexcept;
        TString ToString() const noexcept;

        inline const NCommon::TIpAddress& ToProto() const noexcept {
            return Address;
        }

    private:
        inline TIpAddress(ui8 family) noexcept
            : Address(family, 0, 0)
        {
        }

        const NCommon::TIpAddress Address;
    };

    inline TStringBuf ToStringBuf(const flatbuffers::FlatBufferBuilder& builder) noexcept {
        return {reinterpret_cast<const char*>(builder.GetBufferPointer()), builder.GetSize()};
    }
}
