#include <infra/netmon/library/iam.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/library/requester.h>
#include <infra/netmon/library/scheduler.h>
#include <infra/netmon/library/settings.h>

#include <library/cpp/json/json_reader.h>

namespace NNetmon {
    class TIamUpdater::TImpl: public TScheduledTask {
    public:
        TImpl()
            : TScheduledTask(TDuration::Hours(1))
        {
        }

        TThreadPool::TFuture Run() override {
            // If IamTokenUrl is set, we expect it to point to the VM metadata service:
            // https://cloud.yandex.ru/docs/compute/operations/vm-info/get-info#gce-metadata
            return THttpRequester::Get()->MakeRequest(
                TLibrarySettings::Get()->GetIamTokenUrl(),
                {"Metadata-Flavor: Google"}
            ).Apply([this](const THttpRequester::TFuture& future) {
                try {
                    TString data = future.GetValue()->Data;
                    NJson::TJsonValue json;
                    NJson::ReadJsonFastTree(data, &json, true);
                    auto token = json["access_token"].GetStringSafe();
                    *IamToken.Own() = token;
                    INFO_LOG << "Updated IAM token" << Endl;
                } catch (...) {
                    ERROR_LOG << "Failed to fetch IAM token from " << TLibrarySettings::Get()->GetIamTokenUrl()
                              << ": " << CurrentExceptionMessage() << Endl;
                    throw;
                }
            });
        };

        TString GetToken() const {
            return *IamToken.Own();
        }
    private:
        TPlainLockedBox<TString> IamToken;
    };

    TIamUpdater::TIamUpdater()
        : Impl(MakeHolder<TImpl>())
        , SchedulerGuard(TLibrarySettings::Get()->GetIamTokenUrl().empty() ? nullptr : Impl->Schedule())
    {
    }

    TIamUpdater::~TIamUpdater() = default;

    TString TIamUpdater::GetToken() const {
        return Impl->GetToken();
    }
}
