#pragma once

#include <infra/netmon/library/metrics.h>

#include <util/memory/pool.h>
#include <util/memory/alloc.h>
#include <util/generic/utility.h>
#include <util/thread/lfstack.h>
#include <util/system/spinlock.h>

namespace NNetmon {
    class TSafeFixedSizeAllocator: public TNonCopyable {
        struct TAlloc {
            inline void* ToPointer() noexcept {
                return this;
            }

            static inline TAlloc* FromPointer(void* ptr) noexcept {
                return (TAlloc*)ptr;
            }

            static constexpr size_t EntitySize(size_t alloc) noexcept {
                return Max(sizeof(TAlloc), alloc);
            }

            static constexpr size_t EntityAlign(size_t align) noexcept {
                return Max(alignof(TAlloc), align);
            }
        };

    public:
        class TFixedGrow: public TMemoryPool::IGrowPolicy {
        public:
            size_t Next(size_t prev) const noexcept override {
                Y_UNUSED(prev);
                // use blocks of fixed size
                return 4 * 1024 * 1024;
            }

            static IGrowPolicy* Instance() noexcept {
                return SingletonWithPriority<TFixedGrow, 0>();
            }
        };

        TSafeFixedSizeAllocator(size_t allocSize, size_t alignSize)
            : Pool_(TAlloc::EntityAlign(alignSize), TFixedGrow::Instance(), TDefaultAllocator::Instance())
            , AlignSize_(TAlloc::EntityAlign(alignSize))
            , AllocSize_(TAlloc::EntitySize(allocSize))
        {
        }

        inline void* Allocate() {
            TAlloc* alloc = nullptr;

            if (!Free_.Dequeue(&alloc)) {
                TGuard<TAdaptiveLock> guard(Lock_);
                TUnistat::Instance().PushSignalUnsafe(ELibrarySignals::MemoryPoolAllocated, AllocSize_);
                return Pool_.Allocate(AllocSize_, AlignSize_);
            }

            return alloc->ToPointer();
        }

        inline void Release(void* ptr) noexcept {
            Free_.Enqueue(TAlloc::FromPointer(ptr));
        }

        inline size_t Size() const noexcept {
            return AllocSize_;
        }

    private:
        TMemoryPool Pool_;
        TAdaptiveLock Lock_;
        const size_t AlignSize_;
        const size_t AllocSize_;
        TLockFreeStack<TAlloc*> Free_;
    };

    template <class T>
    class TSafeSmallObjAllocator: public TNonCopyable {
    public:
        inline TSafeSmallObjAllocator()
            : Alloc_(sizeof(T), alignof(T))
        {
        }

        inline T* Allocate() {
            return (T*)Alloc_.Allocate();
        }

        inline void Release(T* t) noexcept {
            Alloc_.Release(t);
        }

    private:
        TSafeFixedSizeAllocator Alloc_;
    };

    template <class T>
    class TSafeObjectFromPool {
    public:
        struct THeader {
            void* Pool;
            // Can't just use T because THeader must be standard layout type for offsetof to work.
            alignas(T) char Obj[sizeof(T)];
        };
        using TPool = TSafeSmallObjAllocator<THeader>;

        inline void* operator new(size_t, TPool* pool) {
            THeader* ret = pool->Allocate();
            ret->Pool = pool;
            return &ret->Obj;
        }

        inline void operator delete(void* ptr, size_t) noexcept {
            DoDelete(ptr);
        }

        inline void operator delete(void* ptr, TPool*) noexcept {
            /*
             * this delete operator can be called automagically by compiler
             */
            DoDelete(ptr);
        }

    private:
        static inline void DoDelete(void* ptr) noexcept {
            static_assert(std::is_standard_layout<THeader>::value, "offsetof is only defined for standard layout types");
            THeader* header = (THeader*)((char*)ptr - offsetof(THeader, Obj));
            ((TPool*)header->Pool)->Release(header);
        }
    };
}
