#include <infra/netmon/library/mongo.h>
#include <infra/netmon/library/settings.h>
#include <infra/netmon/library/thread_pool.h>

#include <library/cpp/mongo/wrappers.h>
#include <library/cpp/logger/global/global.h>

#include <util/string/builder.h>

namespace NNetmon {
    namespace NMongo {
        namespace {
            static void LogHandler(mongoc_log_level_t level, const char* domain, const char* message, void* userdata) {
                Y_UNUSED(userdata);

            #define LOG_MONGO_MESSAGE_TO(log) log << "Mongo " << domain << ": " <<  message << Endl;
                switch (level) {
                    case MONGOC_LOG_LEVEL_ERROR: {
                        LOG_MONGO_MESSAGE_TO(ERROR_LOG);
                        break;
                    }
                    case MONGOC_LOG_LEVEL_CRITICAL: {
                        LOG_MONGO_MESSAGE_TO(FATAL_LOG);
                        break;
                    }
                    case MONGOC_LOG_LEVEL_WARNING: {
                        LOG_MONGO_MESSAGE_TO(WARNING_LOG);
                        break;
                    }
                    case MONGOC_LOG_LEVEL_MESSAGE: {
                        LOG_MONGO_MESSAGE_TO(NOTICE_LOG);
                        break;
                    }
                    case MONGOC_LOG_LEVEL_INFO: {
                        LOG_MONGO_MESSAGE_TO(INFO_LOG);
                        break;
                    }
                    case MONGOC_LOG_LEVEL_DEBUG: {
                        LOG_MONGO_MESSAGE_TO(DEBUG_LOG);
                        break;
                    }
                    case MONGOC_LOG_LEVEL_TRACE: {
                        LOG_MONGO_MESSAGE_TO(DEBUG_LOG);
                        break;
                    }
                    default: {
                        break;
                    }
                }
            #undef LOG_MONGO_MESSAGE_TO
            }
        }

        class TClient::TImpl: public TNonCopyable {
        public:
            inline TImpl()
                : Helper(Options.GetConnectionString())
            {
            }

            NThreading::TFuture<void> Insert(const TString& collection, const NJson::TJsonValue& value) {
                auto promise(NThreading::NewPromise<void>());
                TExecutor::Get()->AddAndForget([=]() mutable {
                    ::NMongo::TError error;
                    if (!Helper.Insert(Options.Database, collection, value, &error)) {
                        promise.SetException(error.Message);
                    } else {
                        promise.SetValue();
                    }
                });
                return promise;
            }

            NThreading::TFuture<void> Update(const TString& collection, const NJson::TJsonValue& selector, const NJson::TJsonValue& updater) {
                auto promise(NThreading::NewPromise<void>());
                TExecutor::Get()->AddAndForget([=]() mutable {
                    ::NMongo::TError error;
                    if (!Helper.Update(Options.Database, collection, selector, updater, &error)) {
                        promise.SetException(error.Message);
                    } else {
                        promise.SetValue();
                    }
                });
                return promise;
            }

            NThreading::TFuture<void> Upsert(const TString& collection, const NJson::TJsonValue& selector, const NJson::TJsonValue& updater) {
                auto promise(NThreading::NewPromise<void>());
                TExecutor::Get()->AddAndForget([=]() mutable {
                    ::NMongo::TError error;
                    if (!Helper.Upsert(Options.Database, collection, selector, updater, &error)) {
                        promise.SetException(error.Message);
                    } else {
                        promise.SetValue();
                    }
                });
                return promise;
            }

            NThreading::TFuture<void> Remove(const TString& collection, const NJson::TJsonValue& selector) {
                auto promise(NThreading::NewPromise<void>());
                TExecutor::Get()->AddAndForget([=]() mutable {
                    ::NMongo::TError error;
                    if (!Helper.Remove(Options.Database, collection, selector, &error)) {
                        promise.SetException(error.Message);
                    } else {
                        promise.SetValue();
                    }
                });
                return promise;
            }

            NThreading::TFuture<TVector<NJson::TJsonValue>> Find(const TString& collection, const NJson::TJsonValue& selector, size_t skip, size_t limit, const NJson::TJsonValue& fields) {
                auto promise(NThreading::NewPromise<TVector<NJson::TJsonValue>>());
                TExecutor::Get()->AddAndForget([=]() mutable {
                    try {
                        TVector<NJson::TJsonValue> result;
                        Helper.Find(Options.Database, collection, [&result](const ::NMongo::TBsonValue& value) {
                            result.push_back(value.ToJson());
                        }, selector, skip, limit, fields);
                        promise.SetValue(result);
                    } catch (...) {
                        promise.SetException(CurrentExceptionMessage());
                    }
                });
                return promise;
            }

        private:
            struct TInitializer {
                TInitializer() {
                    ::NMongo::Init();
                    ::NMongo::SetLogHandler(LogHandler);
                }

                ~TInitializer() {
                    ::NMongo::Cleanup();
                }
            };

            struct TOptions {
                TOptions()
                    : Database(TLibrarySettings::Get()->GetMongoDatabase())
                    , ReplicaSet(TLibrarySettings::Get()->GetMongoReplicaSet())
                    , ConnectTimeout(TDuration::Seconds(5))
                    , SocketTimeout(TDuration::Seconds(10))
                    , ServerSelectionTimeout(TDuration::Seconds(10))
                {
                }

                inline TString GetConnectionString() const {
                    TStringBuilder builder;
                    builder << TLibrarySettings::Get()->GetMongoUri()
                            << "?connectTimeoutMS=" << ConnectTimeout.MilliSeconds()
                            << "&socketTimeoutMS=" << SocketTimeout.MilliSeconds()
                            << "&serverSelectionTimeoutMS=" << ServerSelectionTimeout.MilliSeconds();
                    if (!ReplicaSet.empty()) {
                        builder << "&replicaSet=" + ReplicaSet;
                    }
                    return builder;
                }

                const TString Database;
                const TString ReplicaSet;
                const TDuration ConnectTimeout;
                const TDuration SocketTimeout;
                const TDuration ServerSelectionTimeout;
            };

            class TExecutor : public TCustomThreadExecutor<TExecutor> {
            public:
                inline TExecutor()
                    : TCustomThreadExecutor("MongoClient")
                {
                }
            };

            TInitializer Initializer;
            TOptions Options;
            ::NMongo::THelper Helper;
        };

        TClient::TClient()
            : Impl(MakeHolder<TImpl>())
        {
        }

        TClient::~TClient() = default;

        NThreading::TFuture<void> TClient::Insert(const TString& collection, const NJson::TJsonValue& value) {
            return Impl->Insert(collection, value);
        }

        NThreading::TFuture<void> TClient::Update(const TString& collection, const NJson::TJsonValue& selector, const NJson::TJsonValue& updater) {
            return Impl->Update(collection, selector, updater);
        }

        NThreading::TFuture<void> TClient::Upsert(const TString& collection, const NJson::TJsonValue& selector, const NJson::TJsonValue& updater) {
            return Impl->Upsert(collection, selector, updater);
        }

        NThreading::TFuture<void> TClient::Remove(const TString& collection, const NJson::TJsonValue& selector) {
            return Impl->Remove(collection, selector);
        }

        NThreading::TFuture<TVector<NJson::TJsonValue>> TClient::Find(const TString& collection, const NJson::TJsonValue& selector, size_t skip, size_t limit, const NJson::TJsonValue& fields) {
            return Impl->Find(collection, selector, skip, limit, fields);
        }
    }
}
