#include <infra/netmon/library/requester.h>
#include <infra/netmon/library/thread_pool.h>
#include <infra/netmon/library/metrics.h>
#include <infra/netmon/library/helpers.h>

#include <library/cpp/http/client/request.h>

#include <util/string/builder.h>

namespace NNetmon {
    namespace {
        inline TBaseThreadExecutor* GetPoolOrDefault(TBaseThreadExecutor* pool = nullptr) {
            if (pool != nullptr) {
                return pool;
            } else {
                return TThreadPool::Get();
            }
        }
    }

    class TNehRequester::TImpl : public INamedThread {
    public:
        TImpl()
            : INamedThread("NehRequester")
            , MultiClient(NNeh::CreateMultiClient())
            , ShutdownPromise(NThreading::NewPromise<NNeh::TResponseRef>())
            , Stopped(false)
        {
            ShutdownPromise.SetException("Server is shutting down");
            Start();
        }

        ~TImpl() {
            AtomicSet(Stopped, 1);

            MultiClient->Interrupt();
            Join();

            auto tasks(TasksBox.Own());
            while (!tasks->Empty()) {
                TTask::TRef task(tasks->PopBack());
                task->Cancel();
            }
        }

        void* ThreadProc() noexcept override {
            NNeh::IMultiClient::TEvent ev;
            while (MultiClient->Wait(ev)) {
                TTask::TRef task(static_cast<TTask*>(ev.UserData));
                {
                    auto tasks(TasksBox.Own());
                    task->Unlink();
                    TUnistat::Instance().PushSignalUnsafe(ELibrarySignals::NehClientParallelRequests, tasks->Size());
                }
                if (ev.Type == NNeh::IMultiClient::TEvent::Response) {
                    NNeh::TResponseRef response(ev.Hndl->Get());
                    if (response->IsError()) {
                        const TString message = TStringBuilder()
                            << "Request to " << response->Request.Addr << " failed with error: "
                            << response->GetErrorText();
                        task->ProcessError(message);
                    } else {
                        task.Release()->ProcessResponse(response);
                    }
                } else if (ev.Type == NNeh::IMultiClient::TEvent::Timeout) {
                    ev.Hndl->Cancel();
                    task->ProcessTimeout();
                } else {
                    Y_ASSERT(0 && "unexpected event");
                }
            }
            return this;
        }

        TFuture MakeRequest(const NNeh::TMessage& message, TInstant deadline, TBaseThreadExecutor* pool) {
            auto tasks(TasksBox.Own());

            if (AtomicGet(Stopped)) {
                return ShutdownPromise.GetFuture();
            }

            TTask::TRef task(MakeHolder<TTask>(this, message, deadline, pool));
            TFuture future_(task->GetFuture());
            tasks->PushBack(task.Release());

            TUnistat::Instance().PushSignalUnsafe(ELibrarySignals::NehClientParallelRequests, tasks->Size());

            return future_;
        }

        void Stop() {
        }

    private:
        class TTask: public TIntrusiveListItem<TTask> {
        public:
            using TRef = THolder<TTask>;
            using TListType = TIntrusiveList<TTask>;

            TTask(TImpl* parent, const NNeh::TMessage& message, TInstant deadline, TBaseThreadExecutor* pool)
                : Parent(parent)
                , Promise(NThreading::NewPromise<NNeh::TResponseRef>())
                , Pool(GetPoolOrDefault(pool))
                , Request(message, deadline, this)
                , Handle(Parent->MultiClient->Request(Request))
            {
            }

            void Cancel() {
                Handle->Cancel();
                Promise.SetException("Server is shutting down");
            }

            void ProcessError(const TString& errorText) noexcept {
                Promise.SetException(errorText);
            }

            void ProcessTimeout() noexcept {
                Promise.SetException("Request timed out");
            }

            void ProcessResponse(NNeh::TResponseRef response) noexcept {
                Pool->AddAndForget([this, response]() {
                    TTask::TRef task(this);
                    Promise.SetValue(response);
                });
            }

            TFuture GetFuture() noexcept {
                return Promise.GetFuture();
            }

        private:
            TImpl* Parent;
            TPromise Promise;
            TBaseThreadExecutor* Pool;

            NNeh::IMultiClient::TRequest Request;
            NNeh::THandleRef Handle;
        };

        NNeh::TMultiClientPtr MultiClient;
        TPromise ShutdownPromise;
        TPlainLockedBox<TTask::TListType> TasksBox;
        TAtomic Stopped;
    };

    TNehRequester::TNehRequester()
        : Impl(MakeHolder<TImpl>())
    {
    }

    TNehRequester::~TNehRequester() {
    }

    TNehRequester::TFuture TNehRequester::MakeRequest(const NNeh::TMessage& msg, TInstant deadline, TBaseThreadExecutor* pool) {
        Y_VERIFY(Impl);
        return Impl->MakeRequest(msg, deadline, pool);
    }

    void TNehRequester::Destroy() {
        Y_VERIFY(Impl);
        Impl.Destroy();
    }

    class THttpRequester::TImpl : public TNonCopyable {
    public:
        TImpl()
            : ShutdownPromise(NThreading::NewPromise<NHttpFetcher::TResultRef>())
            , Stopped(0)
        {
            ShutdownPromise.SetException("Server is shutting down");
        }

        ~TImpl() {
            AtomicSet(Stopped, 0);

            auto tasks(TasksBox.Own());
            while (!tasks->Empty()) {
                TTask::TRef task(tasks->PopBack());
                task->Cancel();
            }
        }

        TFuture MakeRequest(const TString& url,
                            const TVector<TString>& headers,
                            const NHttp::TFetchOptions& options,
                            TBaseThreadExecutor* pool) {
            auto tasks(TasksBox.Own());

            if (AtomicGet(Stopped)) {
                return ShutdownPromise.GetFuture();
            }

            TTask::TRef task(MakeHolder<TTask>(this, url, headers, options, pool));
            TFuture future_(task->GetFuture());
            tasks->PushBack(task.Release());

            TUnistat::Instance().PushSignalUnsafe(ELibrarySignals::HttpClientParallelRequests, tasks->Size());

            return future_;
        }

    private:
        class TTask: public TIntrusiveListItem<TTask> {
        public:
            using TRef = THolder<TTask>;
            using TListType = TIntrusiveList<TTask>;

            TTask(TImpl* parent, const TString& url, const TVector<TString>& headers, const NHttp::TFetchOptions& options, TBaseThreadExecutor* pool)
                : Parent(parent)
                , Promise(NThreading::NewPromise<NHttpFetcher::TResultRef>())
                , Pool(GetPoolOrDefault(pool))
                , Query(url, headers, options)
                , State(Parent->FetchClient.Fetch(Query, std::bind(&TTask::OnComplete, this, std::placeholders::_1)))
            {
            }

            void Cancel() {
                State.Cancel();
                Promise.SetException("Task cancelled");
            }

            TFuture GetFuture() noexcept {
                return Promise.GetFuture();
            }

        private:
            void OnComplete(NHttpFetcher::TResultRef result) noexcept {
                {
                    auto tasks(Parent->TasksBox.Own());
                    Unlink();
                    TUnistat::Instance().PushSignalUnsafe(ELibrarySignals::HttpClientParallelRequests, tasks->Size());
                }
                Pool->AddAndForget([this, result]() {
                    TTask::TRef task(this);
                    if (result->Success()) {
                        task->Promise.SetValue(result);
                    } else {
                        task->Promise.SetException(TStringBuilder() << "Request to " << result->RequestUrl << " failed with code " << result->Code);
                    }
                });
            }

            TImpl* Parent;
            TPromise Promise;
            TBaseThreadExecutor* Pool;

            NHttp::TFetchQuery Query;
            NHttp::TFetchState State;
        };

        NHttp::TFetchClient FetchClient;
        TPromise ShutdownPromise;
        TPlainLockedBox<TTask::TListType> TasksBox;
        TAtomic Stopped;
    };

    THttpRequester::THttpRequester()
        : Impl(MakeHolder<TImpl>())
    {
    }

    THttpRequester::~THttpRequester() {
    }

    THttpRequester::TFuture THttpRequester::MakeRequest(const TString& url,
                                                        const TVector<TString>& headers,
                                                        const NHttp::TFetchOptions& options,
                                                        TBaseThreadExecutor* pool) {
        Y_VERIFY(Impl);
        return Impl->MakeRequest(url, headers, options, pool);
    }

    void THttpRequester::Destroy() {
        Y_VERIFY(Impl);
        Impl.Destroy();
    }
}
