#include <infra/netmon/library/requester.h>
#include <infra/netmon/library/api_handler_helpers.h>

#include <infra/netmon/idl/test.fbs.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

using namespace NNetmon;

namespace {
    struct TRequesterContext : public TNonCopyable {
    };

    class TTestPingReply: public THttpReply<THttpDispatcher, TRequesterContext> {
    public:
        TTestPingReply(TRequesterContext& serverContext, const TServiceRequest::TRef clientContext)
            : THttpReply<THttpDispatcher, TRequesterContext>(serverContext, clientContext)
        {
        }

        void Process() {
            Y_VERIFY(Request->Generated() > 0);
            Builder.Finish(NTest::CreateTTestPingResponse(
                Builder,
                TInstant::Now().Seconds()
            ));
        }

        void PreprocessRequest(THttpInput&) {
        }

        void ParseRequest(THttpInput& input) {
            ReadFlatBuffer(&input, Request);
            Y_VERIFY(Request.Verify());
        }

        void WriteResponse(THttpOutput& output) {
            TString body;
            WriteFlatBuffer(body, Builder);
            output << THttpResponse()
                .SetContentType(TStringBuf("application/x-flatbuf"))
                .SetContent(body);
            output.Finish();
        }

    private:
        TFlatObject<NTest::TTestPingRequest> Request;
        flatbuffers::FlatBufferBuilder Builder;
    };

    class THelper {
    public:
        THelper()
            : Port(PortManager.GetPort(12345))
            , Service(Context)
            , Server(THttpServerOptions().SetHost("localhost").SetPort(Port).SetThreads(1))
        {
            Server.Add("/ping", Service);
        }

        TWebServer::TThreadGuard Run() {
            return TWebServer::TThreadGuard(Server);
        }

        inline TString MakeProperBody() {
            TString body;
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NTest::CreateTTestPingRequest(
                builder,
                TInstant::Now().Seconds()
            ));
            WriteFlatBuffer(body, builder);
            return body;
        }

        inline TString GetProperUrl(const TString& prefix) {
            return TStringBuilder() << prefix << "://localhost:" << Port << "/ping";
        }

        inline TString MakeInvalidBody() {
            return "wrong";
        }

        inline TString GetInvalidUrl(const TString& prefix) {
            return TStringBuilder() << prefix << "://localhost:" << Port << "/ping-not-exists";
        }

    private:
        TPortManager PortManager;
        ui16 Port;

        TRequesterContext Context;
        THttpHandler<TTestPingReply> Service;

        TWebServer Server;
    };
}



class TNehRequesterTest: public TTestBase {
    UNIT_TEST_SUITE(TNehRequesterTest);
    UNIT_TEST(TestRequest)
    UNIT_TEST_SUITE_END();

private:
    inline void TestRequest() {
        THelper helper;
        auto guard(helper.Run());

        {
            NNeh::TMessage message;
            message.Addr = helper.GetProperUrl("post2");
            message.Data = helper.MakeProperBody();

            auto response(TNehRequester::Get()->MakeRequest(message).GetValue(TDuration::Max()));
            TFlatObject<NTest::TTestPingResponse> decodedResponse;
            ReadFlatBuffer(response->Data, decodedResponse);
            UNIT_ASSERT(decodedResponse.Verify());
            UNIT_ASSERT(decodedResponse->Generated() > 0);
        }

        {
            NNeh::TMessage message;
            message.Addr = helper.GetInvalidUrl("post2");
            message.Data = helper.MakeInvalidBody();
            auto future(TNehRequester::Get()->MakeRequest(message));
            future.Wait();
            UNIT_ASSERT(future.HasException());
        }
    }
};

UNIT_TEST_SUITE_REGISTRATION(TNehRequesterTest);

class THttpRequesterTest: public TTestBase {
    UNIT_TEST_SUITE(THttpRequesterTest);
    UNIT_TEST(TestRequest)
    UNIT_TEST_SUITE_END();

private:
    inline void TestRequest() {
        THelper helper;
        auto guard(helper.Run());

        {
            auto response(THttpRequester::Get()->MakeRequest(
                helper.GetProperUrl("http"),
                {},
                NHttp::TFetchOptions().SetPostData(helper.MakeProperBody())
            ).GetValue(TDuration::Max()));
            TFlatObject<NTest::TTestPingResponse> decodedResponse;
            ReadFlatBuffer(response->Data, decodedResponse);
            UNIT_ASSERT(decodedResponse.Verify());
            UNIT_ASSERT(decodedResponse->Generated() > 0);
        }

        {
            auto future(THttpRequester::Get()->MakeRequest(
                helper.GetInvalidUrl("http"),
                {},
                NHttp::TFetchOptions().SetPostData(helper.MakeInvalidBody())
            ));
            future.Wait();
            UNIT_ASSERT(future.HasException());
        }
    }
};

UNIT_TEST_SUITE_REGISTRATION(THttpRequesterTest);
