#include <infra/netmon/library/scheduler.h>
#include <infra/netmon/library/futures.h>
#include <infra/netmon/library/helpers.h>
#include <infra/netmon/library/metrics.h>

#include <util/draft/holder_vector.h>
#include <util/random/random.h>

namespace NNetmon {
    namespace {
        const TDuration MAX_STARTUP_DELAY = TDuration::Seconds(60);

        class TTaskCommand : public ISchedulerCommand {
        public:
            TTaskCommand(TScheduledTask* task, EType command)
                : Task(task)
                , Command(command)
            {
            }

            inline EType Type() override {
                return Command;
            }

            TScheduledTask* Task;
            EType Command;
        };

        class TWaiterCommand : public ISchedulerCommand {
        public:
            TWaiterCommand(const TThreadPool::TPromise& promise)
                : Promise(promise)
            {
            }

            inline EType Type() override {
                return EType::WAITER;
            }

            TThreadPool::TPromise Promise;
        };

        class TCommandVector: public THolderVector<ISchedulerCommand> {
        public:
            using THolderVector<ISchedulerCommand>::THolderVector;

            void push_back(ISchedulerCommand* command) {
                PushBack(command);
            };
        };
    }

    TScheduledTask::TScheduledTask(const TDuration& interval, bool withoutDelay, const TDuration& errorInterval)
        : Interval(interval)
        , ErrorInterval(errorInterval)
    {
        if (Interval && !withoutDelay) {
            Deadline = TDuration::MilliSeconds(
                RandomNumber<ui64>(Min(MAX_STARTUP_DELAY.MilliSeconds(), Interval.MilliSeconds()))
            ).ToDeadLine();
        }
    }

    TScheduledTask::TScheduledTask(const TDuration& interval, const TDuration& startDelay, const TDuration& errorInterval)
        : Interval(interval)
        , ErrorInterval(errorInterval)
        , Deadline(startDelay.ToDeadLine())
    {
    }

    TScheduledTask::TScheduledTask(TScheduledTask::TRoundToIntervalTag,
                                   const TDuration& interval,
                                   const TDuration& offset,
                                   bool firstRunWithoutDelay,
                                   const TDuration& errorInterval)
        : Interval(interval)
        , ErrorInterval(errorInterval)
        , RoundToInterval(true)
        , IntervalOffset(offset)
    {
        if (Interval && !firstRunWithoutDelay) {
            Deadline = CalculateNextDeadline();
        }
    }

    TThreadPool::TFuture TScheduledTask::Start() noexcept {
        auto state(State.Own());
        if (state->Registered) {
            Deadline = CalculateNextDeadline();
            TaskTimer.Reset();
            state->Future = Run();
            DEBUG_LOG << "Task '" << GetName() << "' started, #" << (ui64)this << Endl;
            return state->Future;
        } else {
            return NThreading::MakeFuture();
        }
    }

    void TScheduledTask::Register() noexcept {
        DEBUG_LOG << "Register task " << GetName() << ", #" << (ui64)this << ", run every " << Interval << Endl;
        {
            auto state(State.Own());
            Y_VERIFY(!state->Registered);
            state->Registered = true;
        }
        TScheduler::Get()->Register(*this);
    }

    TThreadPool::TFuture TScheduledTask::Delete() noexcept {
        DEBUG_LOG << "Deleting task " << GetName() << ", #" << (ui64)this << Endl;
        {
            auto state(State.Own());
            Y_VERIFY(state->Registered);
            state->Registered = false;
        }
        return TScheduler::Get()->Delete(*this);
    }

    void TScheduledTask::UnlinkFromScheduler() noexcept {
        TAvlTreeItem<TScheduledTask, TCompareUsingDeadline>::Unlink();
        TIntrusiveListItem<TScheduledTask>::Unlink();
    }

    bool TScheduledTask::Spin() noexcept {
        if (State.Own()->Registered) {
            TScheduler::Get()->Spin(*this);
            return true;
        } else {
            return false;
        }
    }

    TThreadPool::TFuture TScheduledTask::SpinAndWait() noexcept {
        if (Spin()) {
            return FutureChain(TScheduler::Get()->CreateWaiter(), [this](){ return GetFuture(); });
        } else {
            // don't wait forever
            return NThreading::MakeFuture();
        }
    }

    void TScheduledTask::Add(TScheduledTask::TTree& tasks) {
        {
            auto state(State.Own());
            state->Running = false;
            if (!state->Registered) {
                UnlinkFromScheduler();
                return;
            }
        }

        TIntrusiveListItem<TScheduledTask>::Unlink();
        tasks.Insert(this);
    }

    void TScheduledTask::Add(TScheduledTask::TListType& tasks) {
        {
            auto state(State.Own());
            if (!state->Registered) {
                UnlinkFromScheduler();
                return;
            }

            Y_VERIFY(!state->Running);
            state->Running = true;
        }

        TAvlTreeItem<TScheduledTask, TCompareUsingDeadline>::Unlink();
        tasks.PushBack(this);
    }

    void TScheduledTask::Reschedule(TScheduledTask::TTree& tasks) {
        auto state(State.Own());
        if (!state->Registered) {
            UnlinkFromScheduler();
            return;
        }

        if (state->Running) {
            Deadline = TInstant::Zero();
        } else {
            TAvlTreeItem<TScheduledTask, TCompareUsingDeadline>::Unlink();
            Deadline = TInstant::Zero();
            tasks.Insert(this);
        }
    }

    void TScheduledTask::SetDeadline(const TInstant& deadline) {
        Deadline = deadline;
    }

    TInstant TScheduledTask::CalculateNextDeadline() const {
        if (RoundToInterval) {
            auto now = TInstant::Now();
            auto deadline = RoundInstant(now, Interval) + IntervalOffset;
            while (deadline <= now) {
                deadline += Interval;
            }
            return deadline;
        } else {
            return Interval.ToDeadLine();
        }
    }

    TScheduler::TScheduler()
        : TThreadedLoop("Scheduler", TDuration::MilliSeconds(10))
    {
        Start();
    }

    TThreadPool::TFuture TScheduler::CreateWaiter() noexcept {
        auto result(NThreading::NewPromise<void>());
        // this will copy our future, but this is ok
        THolder<ISchedulerCommand> command(MakeHolder<TWaiterCommand>(result));
        ControlQueue.Enqueue(command.Release());
        return result;
    }

    void TScheduler::Register(TScheduledTask& task) noexcept {
        THolder<ISchedulerCommand> command(MakeHolder<TTaskCommand>(&task, ISchedulerCommand::REGISTER));
        ControlQueue.Enqueue(command.Release());
    }

    TThreadPool::TFuture TScheduler::Delete(TScheduledTask& task) noexcept {
        THolder<ISchedulerCommand> command(MakeHolder<TTaskCommand>(&task, ISchedulerCommand::DELETE));
        ControlQueue.Enqueue(command.Release());
        // scheduler may still use task, wait for one loop
        if (!task.GetFuture().HasException()) {
            return NThreading::WaitExceptionOrAll(task.GetFuture(), CreateWaiter());
        } else {
            return CreateWaiter();
        }
    }

    void TScheduler::Spin(TScheduledTask& task) noexcept {
        THolder<ISchedulerCommand> command(MakeHolder<TTaskCommand>(&task, ISchedulerCommand::SPIN));
        ControlQueue.Enqueue(command.Release());
    }

    TScheduler::~TScheduler() {
        ScheduleStop();
        Join();
        // all task must be removed before scheduler stop
        Y_VERIFY(RunningTasks.Empty());
        Y_VERIFY(ScheduledTasks.Empty());
    }

    void TScheduler::OnLoop() {
        TUnistatTimer timer{TUnistat::Instance(), ELibrarySignals::SchedulerIterationTime};

        TInstant now(TInstant::Now());

        TGuard<TAdaptiveLock> guard(TaskLock);

        TCommandVector commands;
        ControlQueue.DequeueAllSingleConsumer(&commands);

        TVector<TThreadPool::TPromise> waiters;
        for (auto& command : commands) {
            switch (command->Type()) {
                case ISchedulerCommand::REGISTER: {
                    dynamic_cast<TTaskCommand*>(command)->Task->Add(ScheduledTasks);
                    break;
                };
                case ISchedulerCommand::SPIN: {
                    dynamic_cast<TTaskCommand*>(command)->Task->Reschedule(ScheduledTasks);
                    break;
                };
                case ISchedulerCommand::DELETE: {
                    dynamic_cast<TTaskCommand*>(command)->Task->UnlinkFromScheduler();
                    break;
                };
                case ISchedulerCommand::WAITER: {
                    waiters.emplace_back(std::move(dynamic_cast<TWaiterCommand*>(command)->Promise));
                    break;
                };
            }
        }

        TVector<TScheduledTask*> tasksToRun;
        while (!ScheduledTasks.Empty()) {
            TScheduledTask* task(&(*ScheduledTasks.First()));
            if (task->GetDeadline() <= now) {
                task->Add(RunningTasks);
                tasksToRun.emplace_back(task);
            } else {
                break;
            }
        }

        // future can execute our callback right away if result is ready, so let's release lock
        guard.Release();
        for (const auto task : tasksToRun) {
            task->Start().Subscribe([task, this](const TThreadPool::TFuture& future_) {
                try {
                    future_.GetValue();
                } catch (...) {
                    task->SetDeadline(task->GetErrorInterval().ToDeadLine());
                    ERROR_LOG << "Task '" << task->GetName() << "' failed with exception, "
                              << CurrentExceptionMessage() << ", took " << task->GetTimer().Get() << Endl;
                }
                TGuard<TAdaptiveLock> guard(TaskLock);
                task->Add(ScheduledTasks);
            });
        }

        for (auto& waiter : waiters) {
            waiter.SetValue();
        }
    }

    void TScheduler::StopTasks() {
        TGuard<TAdaptiveLock> guard(TaskLock);
        for (auto& task : RunningTasks) {
            task.Stop();
        }
    }
}
