#pragma once

#include <infra/netmon/library/thread_pool.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/library/helpers.h>

#include <library/cpp/containers/intrusive_avl_tree/avltree.h>

#include <util/generic/intrlist.h>
#include <util/datetime/cputimer.h>
#include <util/system/type_name.h>

namespace NNetmon {
    class TScheduler;

    class ISchedulerCommand {
    public:
        enum EType {
            REGISTER,
            SPIN,
            DELETE,
            WAITER
        };

        virtual ~ISchedulerCommand() = default;
        virtual EType Type() = 0;
    };

    class TScheduledTask: public TAvlTreeItem<TScheduledTask, TCompareUsingDeadline>,
                          public TIntrusiveListItem<TScheduledTask> {
    public:
        friend TScheduler;
        friend TCompareUsingDeadline;

        using TListType = TIntrusiveList<TScheduledTask>;
        using TTree = TAvlTree<TScheduledTask, TCompareUsingDeadline>;

        static constexpr TDuration DefaultErrorInterval = TDuration::Seconds(5);

        class TTaskGuard: public TNonCopyable {
        public:
            inline TTaskGuard() noexcept
                : TTaskGuard(nullptr)
            {
            }

            inline TTaskGuard(TScheduledTask* task) noexcept
                : Task(task)
            {
                if (Task) {
                    Task->Register();
                }
            }

            inline TTaskGuard(TTaskGuard&& other) noexcept
                : Task(other.Task)
            {
                other.Task = nullptr;
            }

            ~TTaskGuard() {
                if (Task) {
                    Task->Delete().Wait();
                }
            }

        private:
            TScheduledTask* Task;
        };

        TScheduledTask(const TDuration& interval, bool withoutDelay=false, const TDuration& errorInterval = DefaultErrorInterval);
        TScheduledTask(const TDuration& interval, const TDuration& startDelay, const TDuration& errorInterval = DefaultErrorInterval);

        // Create a task to be run at time points that are multiples of interval.
        // This prevents drift caused by errorInterval and other tasks.
        struct TRoundToIntervalTag {};
        TScheduledTask(TRoundToIntervalTag,
                       const TDuration& interval,
                       const TDuration& offset,
                       bool firstRunWithoutDelay = false,
                       const TDuration& errorInterval = DefaultErrorInterval);

        virtual ~TScheduledTask() = default;

        virtual TThreadPool::TFuture Run() = 0;

        bool Spin() noexcept;
        [[nodiscard]] TThreadPool::TFuture SpinAndWait() noexcept;

        [[nodiscard]] inline TTaskGuard Schedule() noexcept {
            return {this};
        }

        inline const TDuration& GetInterval() const noexcept {
            return Interval;
        }
        inline const TDuration& GetErrorInterval() const noexcept {
            return ErrorInterval;
        }
        inline const TInstant& GetDeadline() const noexcept {
            return Deadline;
        }

        TAtomic& ShouldStop() noexcept {
            return ShouldStop_;
        }

    private:
        TThreadPool::TFuture Start() noexcept;
        inline void Stop() noexcept {
            AtomicSet(ShouldStop_, true);
        }

        void Register() noexcept;
        [[nodiscard]] TThreadPool::TFuture Delete() noexcept;
        void UnlinkFromScheduler() noexcept;

        void Add(TScheduledTask::TTree& tasks);
        void Add(TScheduledTask::TListType& tasks);

        void Reschedule(TScheduledTask::TTree& tasks);
        void SetDeadline(const TInstant& deadline);

        TInstant CalculateNextDeadline() const;

        [[nodiscard]] inline TThreadPool::TFuture GetFuture() const noexcept {
            return State.Own()->Future;
        }

        inline const TSimpleTimer& GetTimer() const noexcept {
            return TaskTimer;
        }
        inline TString GetName() const noexcept {
            return TypeName(*this);
        }

        struct TTaskState {
            inline TTaskState()
                : Running(false)
                , Registered(false)
                , Future(NThreading::MakeFuture())
            {
            }

            bool Running;
            bool Registered;
            TThreadPool::TFuture Future;
        };

        const TDuration Interval;
        const TDuration ErrorInterval;

        const bool RoundToInterval = false;
        const TDuration IntervalOffset;

        TPlainLockedBox<TTaskState> State;
        TAtomic ShouldStop_ = false;

        TInstant Deadline;
        TSimpleTimer TaskTimer;
    };

    class TScheduler: public TThreadedLoop {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        friend TScheduledTask;

        ~TScheduler();

        void OnLoop() override;

        static TScheduler* Get() {
            return SingletonWithPriority<TScheduler, 100002>();
        }

        [[nodiscard]] TThreadPool::TFuture CreateWaiter() noexcept;

        void StopTasks();

    private:
        TScheduler();

        void Register(TScheduledTask& task) noexcept;
        void Spin(TScheduledTask& task) noexcept;
        [[nodiscard]] TThreadPool::TFuture Delete(TScheduledTask& task) noexcept;

        TAdaptiveLock TaskLock;
        TScheduledTask::TTree ScheduledTasks;
        TScheduledTask::TListType RunningTasks;

        TLockFreeStack<ISchedulerCommand*> ControlQueue;
    };
}
