#include <infra/netmon/library/settings.h>

#include <library/cpp/config/config.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <util/folder/path.h>
#include <util/generic/algorithm.h>
#include <util/generic/xrange.h>
#include <util/string/builder.h>
#include <util/system/info.h>

namespace NNetmon {
    TLibrarySettings::TLibrarySettings()
        : HttpPort(14242)
        , HttpMaxQueueSize(32768)
        , HttpMaxConnections(32768)
        , HttpListenBacklog(16384)
        , HttpClientTimeout(TDuration::Seconds(10))

        , InterconnectPort(14243)

        , ClickHouseUseCompression(false)
        , ClickhousePoolSize(8UL)

        , BlackboxUrl("http://blackbox.yandex-team.ru")

        , ZookeeperRoot("/netmon")

        , MongoDatabase("netmon")

        , TvmAddress("localhost")
        , TvmPort(18080)
        , TvmAuthToken("")
        , TvmNetmonAlias("netmon")
        , TvmBlackboxAlias("blackbox-yateam")

        , RequestLogsEnabled(false)
    {
        SetCpuGuarantee(NSystemInfo::NumberOfCpus());
    }

    void TLibrarySettings::Load(const NConfig::TConfig& config) {
        SetCpuGuarantee(config["cpu_guarantee"].As<ui32>(CpuGuarantee));

        ClickHouseShards.clear();
        if (!config["clickhouse"].IsNull()) {
            const auto& clickhouse = config["clickhouse"];

            ClickHouseUsername = clickhouse["username"].As<TString>("");
            ClickHousePassword = clickhouse["password"].As<TString>("");
            ClickHouseUseCompression = clickhouse["compress"].As<bool>(false);

            TryToParseShards(clickhouse["shards"], ClickHouseShards);
        }

        BlackboxUrl = config["blackbox_url"].As<TString>(BlackboxUrl);

        if (!config["cloud"].IsNull()) {
            const auto& cloudConfig = config["cloud"];
            IamTokenUrl = cloudConfig["iam_token_url"].As<TString>(IamTokenUrl);
        }

        if (!config["zookeeper"].IsNull()) {
            const auto& zookeeper = config["zookeeper"];

            ZookeeperRoot = zookeeper["prefix"].As<TString>(ZookeeperRoot);
            ZookeeperAddress = zookeeper["hosts"].As<TString>(ZookeeperAddress);
        }

        if (!config["mongo"].IsNull()) {
            const auto& mongoConfig = config["mongo"];

            MongoUsername = mongoConfig["username"].As<TString>(MongoUsername);
            MongoPassword = mongoConfig["password"].As<TString>(MongoPassword);
            MongoAddress = mongoConfig["hosts"].As<TString>(MongoAddress);
            MongoDatabase = mongoConfig["database"].As<TString>(MongoDatabase);
            MongoReplicaSet = mongoConfig["replica_set"].As<TString>(MongoReplicaSet);
        }

        if (!config["tvm"].IsNull()) {
            const auto& tvmConfig = config["tvm"];

            TvmAddress = tvmConfig["host"].As<TString>(TvmAddress);
            TvmPort = tvmConfig["port"].As<ui16>(TvmPort);
            TvmAuthToken = tvmConfig["token"].As<TString>(TvmAuthToken);
            TvmNetmonAlias = tvmConfig["netmon_alias"].As<TString>(TvmNetmonAlias);
            TvmBlackboxAlias = tvmConfig["blackbox_alias"].As<TString>(TvmBlackboxAlias);
        }
        
        if (!config["http"].IsNull()) {
            const auto& httpConfig = config["http"];

            HttpMaxQueueSize = httpConfig["max_queue_size"].As<ui32>(HttpMaxQueueSize);
            HttpMaxConnections = httpConfig["max_connections"].As<ui32>(HttpMaxConnections);
            HttpListenBacklog = httpConfig["listen_backlog"].As<int>(HttpListenBacklog);
            TDuration::TryParse(httpConfig["client_timeout"].As<TString>(), HttpClientTimeout);
        }
    }

    TString TLibrarySettings::GetZookeeperUri() const {
        if (ZookeeperAddress.empty()) {
            return "";
        } else {
            return TStringBuilder()
                << GetZookeeperAddress()
                << TFsPath("/" + GetZookeeperRoot()).Fix().GetPath();
        }
    }

    TString TLibrarySettings::GetMongoUri() const {
        if (MongoAddress.empty()) {
            return "";
        } else {
            return TStringBuilder()
                << "mongodb://"
                << GetMongoUsername() << ":" << GetMongoPassword() << "@"
                << GetMongoAddress()
                << TFsPath("/" + GetMongoDatabase()).Fix().GetPath();
        }
    }

    bool TLibrarySettings::TryToParseShards(const NConfig::TConfig& config, TShardList& shards) {
        if (!config.IsNull() && config.IsA<NConfig::TArray>()) {
            for (const auto& idx : xrange(config.GetArraySize())) {
                TLibrarySettings::TShard el{
                    config[idx]["host"].As<TString>(),
                    config[idx]["port"].As<ui64>(),
                    config[idx]["index"].As<TString>(""),
                };
                shards.push_back(el);
            }
            StableSortBy(shards.begin(), shards.end(), [](const auto& x) {
                return x.Index;
            });
            return true;
        } else {
            return false;
        }
    }
}
