#pragma once

#include <library/cpp/threading/future/future.h>
#include <library/cpp/threading/work_stealing/queue.h>

#include <util/thread/pool.h>

namespace NNetmon {
    class TOffloadedTask: public TMoveOnly {
    public:
        using TRef = THolder<TOffloadedTask>;

        virtual ~TOffloadedTask() = default;

        virtual void operator()() = 0;
    };

    class TBaseThreadExecutor: public TNonCopyable {
    public:
        using TFuture = NThreading::TFuture<void>;
        using TPromise = NThreading::TPromise<void>;

        TBaseThreadExecutor(const TString& poolName, size_t poolSize=0);

        virtual ~TBaseThreadExecutor();

        [[nodiscard]] TFuture Add(TOffloadedTask::TRef&& task, bool major = true);
        [[nodiscard]] TFuture Add(const std::function<void()>& func, bool major = true);
        void AddAndForget(const std::function<void()>& func, bool major = true);

        inline std::size_t Size() const noexcept {
            return Size_;
        }

        void Stop();

    private:
        void Start(const TString& poolName);

        THolder<TWorkStealingMtpQueue> Queue_;
        const size_t Size_;
    };

    template <class TPool, size_t PoolSize=0>
    class TCustomThreadExecutor: public TBaseThreadExecutor {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        static TPool* Get() {
            return SingletonWithPriority<TPool, 100001>();
        }

    protected:
        TCustomThreadExecutor(const TString& poolName, size_t poolSize=PoolSize)
            : TBaseThreadExecutor(poolName, poolSize)
        {
        }
    };

    class TThreadPool: public TCustomThreadExecutor<TThreadPool> {
    public:
        TThreadPool()
            : TCustomThreadExecutor("ThreadPool")
        {
        }
    };

    // thread that serializes heavy tasks for reduce CPU usage
    class THeavyTaskThread: public TCustomThreadExecutor<THeavyTaskThread, 1> {
    public:
        THeavyTaskThread()
            : TCustomThreadExecutor("HeavyTaskThread")
        {
        }
    };
}
