#include <infra/netmon/library/api_handler_helpers.h>
#include <infra/netmon/library/web_server.h>

#include <library/cpp/http/io/stream.h>
#include <library/cpp/http/server/response.h>

namespace NNetmon {
    namespace {
        THolder<THttpServer> StartHttp(
                const THttpServerOptions& options,
                THttpServer::ICallBack* callBack,
                THttpServer::TMtpQueueRef processingQueue) {
            class TFailedHttpRequestMtpQueue: public ::TThreadPool {
                void Start(size_t, size_t) override {
                    // always start with 1 thread without limit
                    TThreadPool::Start(1, 0);
                }
            };
            THttpServer::TMtpQueueRef failedQueue(new TFailedHttpRequestMtpQueue());
            THolder<THttpServer> httpServer(new THttpServer(callBack, processingQueue, failedQueue, options));

            if (!httpServer->Start()) {
                ythrow yexception() << "web server fail to start: " << httpServer->GetError();
            }

            return httpServer;
        }
    }

    TWebReplier::TWebReplier(TWebServer* server)
        : Server(server)
        , Ready(false)
    {
    }

    bool TWebReplier::DoReply(const TRequestReplier::TReplyParams& params) {
        if (Ready) {
            return true;
        }
        const TParsedHttpFull parsedLine(params.Input.FirstLine());
        IServiceReplier* serviceReplier = Server->GetReplier(parsedLine);

        if (TLibrarySettings::Get()->AreRequestLogsEnabled()) {
            auto socket = TClientRequest::Socket();
            NAddr::TOpaqueAddr requesterAddr;
            TString clientAddr;

            if (!getpeername(socket, requesterAddr.MutableAddr(), requesterAddr.LenPtr())) {
                clientAddr = NAddr::PrintHost(requesterAddr);
            }

            const auto credentials(ExtractCredentials(params.Input.Headers()));

            INFO_LOG << "Request " << params.Input.FirstLine() << " from " << credentials.UserAgent << " " << clientAddr << Endl;
        }

        if (serviceReplier == nullptr) {
            params.Output << THttpResponse(HTTP_NOT_FOUND).SetContent("Service not found!");
            return true;
        } else {
            TServiceRequest::TRef request(MakeAtomicShared<TServiceRequest>(params));
            serviceReplier->DoReply(request);
            request->GetFuture().Subscribe([this](const TServiceRequest::TFuture&) {
                try {
                    Finish();
                } catch(...) {
                    ERROR_LOG << "Can't finish request: " << CurrentExceptionMessage() << Endl;
                }
            });
            return false;
        }
    }

    TWebServer::TWebServer(const THttpServerOptions& options)
        : INamedThread("WebServer")
        , HttpOptions(options)
        , ProcessingQueue(new TWorkStealingMtpQueue())
    {
        ProcessingQueue->Init(HttpOptions.MaxQueueSize);
    }

    TWebServer::~TWebServer() {
        HttpServer.Reset();
        ProcessingQueue.Reset();
    }

    void TWebServer::Start() {
        Y_VERIFY(!HttpServer);
        HttpServer.Reset(StartHttp(HttpOptions, this, ProcessingQueue));
        INamedThread::Start();
    }

    TClientRequest* TWebServer::CreateClient() {
        return new TWebReplier(this);
    }

    void* TWebServer::ThreadProc() noexcept {
        Y_VERIFY(HttpServer);
        try {
            HttpServer->Wait();
        } catch (...) {
            ERROR_LOG << "Unhandled exception: " << CurrentExceptionMessage() << Endl;
        }
        return this;
    }

    void TWebServer::Stop() {
        if (HttpServer) {
            HttpServer->Shutdown();
            Join();
        }
    }

    void TWebServer::Add(const TString& path, IServiceReplier& service) {
        Services[path] = &service;
    }

    IServiceReplier* TWebServer::GetReplier(const TParsedHttpFull& parsedHttp) {
        auto it = Services.find(parsedHttp.Path);
        return it.IsEnd() ? nullptr : it->second;
    }
}
