#pragma once

#include <infra/netmon/library/helpers.h>
#include <infra/netmon/metrics.h>

#include <util/generic/hash.h>

#include <library/cpp/threading/future/future.h>
#include <library/cpp/threading/work_stealing/queue.h>
#include <library/cpp/http/server/http.h>
#include <library/cpp/http/misc/parsed_request.h>

namespace NNetmon {
    class TWebServer;

    class TServiceRequest: public TNonCopyable {
    public:
        using TFuture = NThreading::TFuture<void>;
        using TPromise = NThreading::TPromise<void>;
        using TRef = TAtomicSharedPtr<TServiceRequest>;

        TServiceRequest(const TRequestReplier::TReplyParams& params)
            : RequestParams(params)
            , Promise(NThreading::NewPromise<void>())
        {
        }

        ~TServiceRequest() {
            // don't forget about undispatched requests
            Finish();
        }

        TFuture GetFuture() {
            return Promise.GetFuture();
        }
        THttpInput& GetInput() const {
            return RequestParams.Input;
        }
        THttpOutput& GetOutput() const {
            return RequestParams.Output;
        }
        void Finish() const {
            if (!Promise.HasValue()) {
                Promise.SetValue();
            }
        }

    private:
        const TRequestReplier::TReplyParams RequestParams;
        mutable TPromise Promise;
    };

    class IServiceReplier: public TNonCopyable {
    public:
        virtual ~IServiceReplier() {
        }

        virtual void DoReply(const TServiceRequest::TRef request) = 0;
    };

    class TWebReplier: public TRequestReplier {
    public:
        TWebReplier(TWebServer* server);

        bool DoReply(const TRequestReplier::TReplyParams& params) override;

    private:
        TWebServer* Server;
        volatile bool Ready;

        // this will be destroyed after call
        inline void Finish() {
            Y_VERIFY(!Ready);
            Ready = true;
            static_cast<IObjectInQueue*>(this)->Process(nullptr);
        }
    };

    class TWebServer: public INamedThread, public THttpServer::ICallBack {
    public:
        using TThreadGuard = TThreadGuard<TWebServer>;

        TWebServer(const THttpServerOptions& options);
        ~TWebServer();

        // implements THttpServer::ICallBack
        TClientRequest* CreateClient() override;

        void OnFailRequest(int) override {
            TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::HttpServerFailedRequests, 1);
        }
        void OnFailRequestEx(const TFailLogData&) override {
            TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::HttpServerFailedRequests, 1);
        }
        void OnException() override {
            TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::HttpServerUncaughtExceptions, 1);
        }
        void OnMaxConn() override {
            TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::HttpServerMaxConnErrors, 1);
        }

        void* ThreadProc() noexcept override;
        void Start();
        void Stop();

        void Add(const TString& path, IServiceReplier& service);

        IServiceReplier* GetReplier(const TParsedHttpFull& parsedHttp);

    private:
        THttpServerOptions HttpOptions;
        TSimpleSharedPtr<TWorkStealingMtpQueue> ProcessingQueue;
        THolder<THttpServer> HttpServer;
        THashMap<TString, IServiceReplier*> Services;
    };
}
