#include <infra/netmon/library/zookeeper.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/library/settings.h>

#include <util/folder/path.h>
#include <util/generic/xrange.h>
#include <util/generic/set.h>
#include <library/cpp/deprecated/split/delim_string_iter.h>

#include <contrib/libs/zookeeper/include/zookeeper.h>

namespace NNetmon {
    namespace NAsyncZookeeper {
        using NZooKeeper::ACLS_ALL;
        using NZooKeeper::TACL;

        namespace {
            const struct ACL_vector DEFAULT_ACLS {
                .count = (i32)ACLS_ALL.size(),
                .data = const_cast<TACL*>(&ACLS_ALL.front())
            };
        }

        TOptions::TOptions(const TString& address)
            : Address(address)
            , Timeout(TDuration::Seconds(30))
            , LogLevel(ELogLevel::LL_ERROR)
        {
        }

        template <class T, class TResult>
        class TAbstractCallback: public TAtomicRefCount<T>, public TNonCopyable {
        public:
            using TRef = TIntrusivePtr<T>;
            using TFuture = typename TResult::TFuture;

            inline TAbstractCallback()
                : Promise(NThreading::NewPromise<TResult>())
            {
            }

            virtual ~TAbstractCallback() = default;

            inline TFuture Execute(zhandle_t* zhandle) noexcept {
                if (zhandle != nullptr) {
                    this->Ref();
                    auto rc(RealExecute(zhandle));
                    if (CheckError(rc)) {
                        this->DecRef();
                    }
                } else {
                    Error(INVALID_STATE);
                }
                return Promise.GetFuture();
            }

        protected:
            virtual inline int RealExecute(zhandle_t* zhandle) noexcept = 0;

            static inline TRef FromVoid(const void* context) noexcept {
                TRef self(static_cast<T*>(const_cast<void*>(context)));
                self->DecRef();
                return self;
            }

            inline bool CheckError(int rc) noexcept {
                switch (rc) {
                    case ZOK: {
                        return false;
                    }
                    case ZNODEEXISTS: {
                        Error(NODE_EXISTS);
                        return true;
                    }
                    case ZNONODE: {
                        Error(NODE_NOT_EXISTS);
                        return true;
                    }
                    case ZBADVERSION: {
                        Error(NODE_BAD_VERSION);
                        return true;
                    }
                    default: {
                        Error(INVALID_STATE);
                        return true;
                    }
                };
            }

            template <typename... Args>
            inline void Success(Args&&... args) noexcept {
                Promise.SetValue(TResult::Success(std::forward<Args>(args)...));
            }

            inline void Error(EResultCode rc) noexcept {
                Promise.SetValue(TResult::Error(rc));
            }

        private:
            mutable typename TResult::TPromise Promise;
        };

        class TBatchCallback: public TAbstractCallback<TBatchCallback, TVoidResult>, public IBatchVisitor {
        public:
            using TAbstractCallback::TAbstractCallback;

            void CreateOperation(const TString& path, const TString& data, ECreateMode createMode, TBuffer& resultingPath) override {
                auto& op(Operations.emplace_back());

                zoo_create_op_init(
                    &op, path.c_str(),
                    data.data(), data.size(),
                    &DEFAULT_ACLS, createMode,
                    resultingPath.Data(), resultingPath.Size());
            }

            void DeleteOperation(const TString& path, int version) override {
                auto& op(Operations.emplace_back());

                zoo_delete_op_init(&op, path.c_str(), version);
            }

            void SetOperation(const TString& path, const TString& data, int version, TStat& stat) override {
                auto& op(Operations.emplace_back());

                zoo_set_op_init(&op, path.c_str(), data.data(), data.size(), version, &stat);
            }

            void CheckOperation(const TString& path, int version) override {
                auto& op(Operations.emplace_back());

                zoo_check_op_init(&op, path.c_str(), version);
            }

        protected:
            inline int RealExecute(zhandle_t* zhandle) noexcept override {
                Results.resize(Operations.size());
                return zoo_amulti(zhandle, Operations.size(), Operations.data(), Results.data(), Delegate, this);
            }

            static void Delegate(int rc, const void* context) noexcept {
                TRef self(FromVoid(context));
                if (!self->CheckError(rc)) {
                    self->Success();
                }
            }

            std::vector<zoo_op_t> Operations;
            std::vector<zoo_op_result_t> Results;
        };

        class TDataCallback: public TAbstractCallback<TDataCallback, TDataResult> {
        public:
            inline TDataCallback(const TString& path, bool watch)
                : Path(path)
                , Watch(watch)
            {
            }

        private:
            inline int RealExecute(zhandle_t* zhandle) noexcept override {
                return zoo_aget(zhandle, Path.c_str(), Watch, Delegate, this);
            }

            static void Delegate(int rc, const char* value, int value_len, const struct Stat* stat, const void* context) noexcept {
                TRef self(FromVoid(context));
                if (!self->CheckError(rc)) {
                    Y_VERIFY(value != nullptr && stat != nullptr);
                    self->Success(TDataTuple(TString(value, value_len), TStat(*stat)));
                }
            }

            const TString Path;
            const bool Watch;
        };

        class TChildrenCallback: public TAbstractCallback<TChildrenCallback, TChildrenResult> {
        public:
            inline TChildrenCallback(const TString& path, bool watch)
                : Path(path)
                , Watch(watch)
            {
            }

        private:
            inline int RealExecute(zhandle_t* zhandle) noexcept override {
                return zoo_aget_children(zhandle, Path.c_str(), Watch, Delegate, this);
            }

            static void Delegate(int rc, const struct String_vector* strings, const void* context) noexcept {
                TRef self(FromVoid(context));
                if (!self->CheckError(rc)) {
                    Y_VERIFY(strings != nullptr);
                    TVector<TString> children;
                    children.reserve(strings->count);
                    for (const auto idx : xrange(strings->count)) {
                        children.emplace_back(strings->data[idx]);
                    }
                    self->Success(std::move(children));
                }
            }

            const TString Path;
            const bool Watch;
        };

        class TExistsCallback: public TAbstractCallback<TExistsCallback, TStatResult> {
        public:
            inline TExistsCallback(const TString& path, bool watch)
                : Path(path)
                , Watch(watch)
            {
            }

        private:
            inline int RealExecute(zhandle_t* zhandle) noexcept override {
                return zoo_aexists(zhandle, Path.c_str(), Watch, Delegate, this);
            }

            static void Delegate(int rc, const struct Stat* stat, const void* context) noexcept {
                TRef self(FromVoid(context));
                if (!self->CheckError(rc)) {
                    Y_VERIFY(stat != nullptr);
                    self->Success(TStat(*stat));
                }
            }

            const TString Path;
            const bool Watch;
        };

        class TSetCallback: public TAbstractCallback<TSetCallback, TStatResult> {
        public:
            inline TSetCallback(const TString& path, const TString& data, int version)
                : Path(path)
                , Data(data)
                , Version(version)
            {
            }

        private:
            inline int RealExecute(zhandle_t* zhandle) noexcept override {
                return zoo_aset(zhandle, Path.c_str(), Data.data(), Data.size(), Version, Delegate, this);
            }

            static void Delegate(int rc, const struct Stat* stat, const void* context) noexcept {
                TRef self(FromVoid(context));
                if (!self->CheckError(rc)) {
                    Y_VERIFY(stat != nullptr);
                    self->Success(TStat(*stat));
                }
            }

            const TString Path;
            const TString Data;
            const int Version;
        };

        class TCreateCallback: public TAbstractCallback<TCreateCallback, TPathResult> {
        public:
            inline TCreateCallback(const TString& path, const TString& data, ECreateMode createMode)
                : Path(path)
                , Data(data)
                , CreateMode(createMode)
            {
            }

        private:
            inline int RealExecute(zhandle_t* zhandle) noexcept override {
                return zoo_acreate(zhandle, Path.c_str(), Data.data(), Data.size(), &DEFAULT_ACLS, CreateMode, Delegate, this);
            }

            static void Delegate(int rc, const char* value, const void* context) noexcept {
                TRef self(FromVoid(context));
                if (!self->CheckError(rc)) {
                    Y_VERIFY(value != nullptr);
                    self->Success(TString(value));
                }
            }

            const TString Path;
            const TString Data;
            const ECreateMode CreateMode;
        };

        class TDeleteCallback: public TAbstractCallback<TDeleteCallback, TVoidResult> {
        public:
            inline TDeleteCallback(const TString& path, int version)
                : Path(path)
                , Version(version)
            {
            }

        private:
            inline int RealExecute(zhandle_t* zhandle) noexcept override {
                return zoo_adelete(zhandle, Path.c_str(), Version, Delegate, this);
            }

            static void Delegate(int rc, const void* context) noexcept {
                TRef self(FromVoid(context));
                if (!self->CheckError(rc)) {
                    self->Success();
                }
            }

            const TString Path;
            const int Version;
        };

        class TClient::TImpl: public TNonCopyable {
        private:
            class TZHandle: public TNonCopyable {
            public:
                TZHandle(TImpl* parent)
                    : Parent(parent)
                {
                    Connect();
                }

                ~TZHandle() {
                    Close();
                }

                inline operator zhandle_t*() noexcept {
                    if (Handle == nullptr || is_unrecoverable(Handle) == ZINVALIDSTATE) {
                        Close();
                        Connect();
                    }
                    return Handle;
                }

            private:
                inline void Connect() {
                    if (Handle != nullptr || Parent->Options.Address.empty()) {
                        return;
                    }

                    zhandle_t* handle = zookeeper_init(
                        Parent->Options.Address.c_str(), WatcherDelegate,
                        Parent->Options.Timeout.MilliSeconds(), nullptr, Parent, 0);
                    if (handle == nullptr) {
                        ERROR_LOG << "Can't connect to zookeeper: " << LastSystemErrorText() << Endl;
                    }

                    Handle = handle;
                }

                inline void Close() noexcept {
                    if (Handle == nullptr) {
                        return;
                    }

                    int rc = zookeeper_close(Handle);
                    if (rc != ZOK) {
                        ERROR_LOG << "Can't destroy zookeeper client: " << zerror(rc) << Endl;
                    }

                    Handle = nullptr;
                }

                TImpl* Parent;
                zhandle_t* Handle = nullptr;
            };

        public:
            inline TImpl(const TOptions& options)
                : Options(options)
                , ZHandleBox(this)
            {
                zoo_set_debug_level((ZooLogLevel)Options.LogLevel);
            }

            inline TImpl()
                : TImpl(TOptions(""))
            {
            }

            inline TVoidResult::TFuture Mutate(const IAsyncOperation::TVectorType& operations) noexcept {
                TBatchCallback::TRef callback(MakeIntrusive<TBatchCallback>());
                for (const auto& op : operations) {
                    op->Prepare(*callback);
                }
                auto box(ZHandleBox.Own());
                return callback->Execute(*box);
            }

            template <class T, typename... Args>
            inline typename T::TFuture Execute(Args&&... args) noexcept {
                typename T::TRef callback(MakeIntrusive<T>(std::forward<Args>(args)...));
                auto box(ZHandleBox.Own());
                return callback->Execute(*box);
            }

            void AddWatcher(const TString& path, IWatcher* watcher) {
                WatcherBox.Own()->Add(path, watcher);
            }

            void RemoveWatcher(const TString& path, IWatcher* watcher) {
                WatcherBox.Own()->Remove(path, watcher);
            }

        private:
            static void WatcherDelegate(zhandle_t*, int type, int state, const char* path, void* context) noexcept {
                static_cast<TImpl*>(context)->OnEvent(type, state, path);
            }

            inline void OnEvent(int type, int state, const char* path) noexcept {
                const TEvent event(static_cast<EEventType>(type), static_cast<EKeeperState>(state), path);
                DEBUG_LOG << "Event of type " << event.EventType << " happened on '" << event.Path << "' (" << event.KeeperState << ")" << Endl;
                if (event.EventType == EEventType::ET_NONE && event.KeeperState == EKeeperState::KS_EXPIRED) {
                    WatcherBox.Own()->OnExpired();
                } else {
                    WatcherBox.Own()->OnEvent(event);
                }
            }

            const TOptions Options;

            class TWatcherIndex {
            public:
                void Add(const TString& path, IWatcher* watcher) {
                    Watchers.emplace(path, watcher);
                    AddUniqueWatcher(watcher);
                }

                void Remove(const TString& path, IWatcher* watcher) {
                    const auto range(Watchers.equal_range(path));
                    for (auto it(range.first); it != range.second; ++it) {
                        if (it->second == watcher) {
                            RemoveUniqueWatcher(watcher);
                            Watchers.erase(it);
                            return;
                        }
                    }
                }

                void OnExpired() {
                    for (auto& pair : UniqueWatchers) {
                        pair.first->OnExpired();
                    }
                }

                void OnEvent(const TEvent& event) {
                    TVector<TString> prefixes{};
                    for (TStringBuf x : TDelimStroka(event.Path, TStringBuf("/"))) {
                        if (prefixes.empty()) {
                            prefixes.emplace_back(x);
                        } else {
                            prefixes.emplace_back(Join(prefixes.back(), x));
                        }
                        const auto range(Watchers.equal_range(TStringBuilder() << "/" << prefixes.back()));
                        for (auto it(range.first); it != range.second; ++it) {
                            it->second->OnEvent(event);
                        }
                    }
                }

            private:
                void AddUniqueWatcher(IWatcher* watcher) {
                    auto it(UniqueWatchers.find(watcher));
                    if (it.IsEnd()) {
                        UniqueWatchers.emplace(watcher, 1);
                    } else {
                        it->second++;
                    }
                }

                void RemoveUniqueWatcher(IWatcher* watcher) {
                    auto it(UniqueWatchers.find(watcher));
                    Y_VERIFY(!it.IsEnd() && it->second > 0);
                    it->second--;
                    if (!it->second) {
                        UniqueWatchers.erase(it);
                    }
                }

                THashMultiMap<TString, IWatcher*> Watchers;
                THashMap<IWatcher*, size_t> UniqueWatchers;
            };
            TPlainLockedBox<TWatcherIndex> WatcherBox;

            TPlainLockedBox<TZHandle> ZHandleBox;
        };

        TClient::TClient()
            : Impl(MakeHolder<TImpl>())
        {
            const auto& uri(TLibrarySettings::Get()->GetZookeeperUri());
            if (uri.empty()) {
                Impl.Reset(MakeHolder<TImpl>());
            } else {
                Impl.Reset(MakeHolder<TImpl>(TOptions(uri)));
            }
        }

        TClient::~TClient() {
        }

        TVoidResult::TFuture TClient::Mutate(const IAsyncOperation::TVectorType& operations) noexcept {
            return Impl->Mutate(operations);
        }

        TDataResult::TFuture TClient::GetData(const TString& path, bool watch) noexcept {
            return Impl->Execute<TDataCallback>(path, watch);
        }

        TChildrenResult::TFuture TClient::GetChildren(const TString& path, bool watch) noexcept {
            return Impl->Execute<TChildrenCallback>(path, watch);
        }

        TStatResult::TFuture TClient::Exists(const TString& path, bool watch) noexcept {
            return Impl->Execute<TExistsCallback>(path, watch);
        }

        TStatResult::TFuture TClient::SetData(const TString& path, const TString& data, int version) noexcept {
            return Impl->Execute<TSetCallback>(path, data, version);
        }

        TPathResult::TFuture TClient::Create(const TString& path, const TString& data, ECreateMode createMode) noexcept {
            return Impl->Execute<TCreateCallback>(path, data, createMode);
        }

        TVoidResult::TFuture TClient::Delete(const TString& path, int version) noexcept {
            return Impl->Execute<TDeleteCallback>(path, version);
        }

        void TClient::AddWatcher(const TString& path, IWatcher* watcher) {
            Impl->AddWatcher(path, watcher);
        }

        void TClient::RemoveWatcher(const TString& path, IWatcher* watcher) {
            Impl->RemoveWatcher(path, watcher);
        }

        template <class T, class TResult>
        class TAbstractRecipe: public TAtomicRefCount<T>, public TNonCopyable {
        public:
            using TRef = TIntrusivePtr<T>;

            inline TAbstractRecipe(TClient& client)
                : Client(client)
                , Promise(NThreading::NewPromise<TResult>())
            {
            }

            virtual ~TAbstractRecipe() = default;

        protected:
            template <typename... Args>
            inline void Success(Args&&... args) noexcept {
                Promise.SetValue(TResult::Success(std::forward<Args>(args)...));
            }

            inline void Error(EResultCode rc) noexcept {
                Promise.SetValue(TResult::Error(rc));
            }

            TClient& Client;
            typename TResult::TPromise Promise;
        };

        class TRecursiveCreateRecipe: public TAbstractRecipe<TRecursiveCreateRecipe, TVoidResult> {
        public:
            inline TRecursiveCreateRecipe(TClient& client, const TString& path)
                : TAbstractRecipe(client)
            {
                TFsPath fs(path);
                fs = fs.Fix();
                while (fs.GetPath() != TStringBuf("/")) {
                    Components.push_back(fs.GetPath());
                    fs = fs.Parent();
                }
            }

            inline TVoidResult::TFuture Execute() noexcept {
                if (Components.empty()) {
                    Success();
                } else {
                    for (auto it(Components.rbegin()); it != Components.rend(); ++it) {
                        Client.Create(*it, "").Subscribe(
                            Y_MEMBER_REF_TO_CB(&TRecursiveCreateRecipe::OnNodeCreated, this));
                    }
                }
                return Promise.GetFuture();
            }

        private:
            void OnNodeCreated(const TPathResult& result) noexcept {
                if (Promise.HasValue()) {
                    return;
                } else if (result.ResultCode() == INVALID_STATE) {
                    Error(INVALID_STATE);
                } else if (++Checked == Components.size()) {
                    Success();
                }
            }

            TVector<TString> Components;
            std::size_t Checked = 0;
        };

        TVoidResult::TFuture TClient::CreateRecursive(const TString& path) noexcept {
            TRecursiveCreateRecipe::TRef operation(MakeIntrusive<TRecursiveCreateRecipe>(*this, path));
            return operation->Execute();
        }

        class TChildrenDataRecipe: public TAbstractRecipe<TChildrenDataRecipe, TChildrenDataResult> {
        public:
            inline TChildrenDataRecipe(TClient& client, const TString& path, bool watchRoot, bool watchChildren)
                : TAbstractRecipe(client)
                , Path(path)
                , WatchRoot(watchRoot)
                , WatchChildren(watchChildren)
            {
            }

            inline TChildrenDataResult::TFuture Execute() noexcept {
                Client.GetChildren(Path, WatchRoot).Subscribe(Y_MEMBER_REF_TO_CB(&TChildrenDataRecipe::OnChildrenList, this));
                return Promise.GetFuture();
            }

        private:
            inline void OnChildrenList(const TChildrenResult& result) noexcept {
                if (result.ResultCode() != OK) {
                    Error(result.ResultCode());
                } else if (result.Result().empty()) {
                    Success(std::move(Result));
                } else {
                    Total = result.Result().size();
                    for (const auto& child : result.Result()) {
                        Client.GetData(Join(Path, child), WatchChildren).Subscribe(Y_MEMBER_REF_TO_CB(&TChildrenDataRecipe::OnChildData, this, child));
                    }
                }
            }

            inline void OnChildData(const TDataResult& result, const TString& path) noexcept {
                if (Promise.HasValue()) {
                    return;
                } else if (!result.Empty()) {
                    Result[path] = result.Result();
                } else if (result.ResultCode() != NODE_NOT_EXISTS) {
                    Error(INVALID_STATE);
                    return;
                }
                if (++Checked == Total) {
                    Success(std::move(Result));
                }
            }

            const TString Path;
            const bool WatchRoot;
            const bool WatchChildren;
            std::size_t Checked = 0;
            std::size_t Total = 0;
            THashMap<TString, TDataTuple> Result;
        };

        TChildrenDataResult::TFuture TClient::GetChildrenData(const TString& path, bool watchRoot, bool watchChildren) noexcept {
            TChildrenDataRecipe::TRef operation(MakeIntrusive<TChildrenDataRecipe>(*this, path, watchRoot, watchChildren));
            return operation->Execute();
        }
    }
}
