#pragma once

#include <infra/netmon/library/zookeeper-impl.h>
#include <infra/netmon/library/helpers.h>

#include <library/cpp/zookeeper/defines.h>
#include <library/cpp/threading/future/future.h>

#include <util/draft/holder_vector.h>
#include <util/generic/buffer.h>
#include <util/generic/hash.h>
#include <util/generic/maybe.h>

#define Y_MEMBER_REF_TO_CB(Member, Pointer, ...) ::NNetmon::NAsyncZookeeper::NImpl::TMemberRefHelper<decltype(Member)>(Member, Pointer, ##__VA_ARGS__)

namespace NNetmon {
    namespace NAsyncZookeeper {
        using NZooKeeper::TStat;
        using NZooKeeper::ELogLevel;
        using NZooKeeper::EEventType;
        using NZooKeeper::EKeeperState;
        using NZooKeeper::ECreateMode;

        template <typename... Args>
        inline TString Join(Args&&... paths) noexcept {
            TStringBuilder builder;
            NImpl::JoinImpl(builder, std::forward<Args>(paths)...);
            return builder;
        }

        struct TOptions {
            TOptions(const TString& address);

            TString Address;
            TDuration Timeout;
            ELogLevel LogLevel;
        };

        struct TEvent {
            EEventType EventType;
            EKeeperState KeeperState;
            TString Path;

            TEvent(EEventType eventType, EKeeperState keeperState, const TString& path)
                : EventType(eventType)
                , KeeperState(keeperState)
                , Path(path)
            {
            }
        };

        class IWatcher {
        public:
            virtual ~IWatcher() = default;
            virtual void OnEvent(const TEvent &event) = 0;
            virtual void OnExpired() = 0;
        };

        enum EResultCode {
            OK,
            NODE_EXISTS,
            NODE_NOT_EXISTS,
            NODE_BAD_VERSION,
            INVALID_STATE
        };

        template <class T>
        class TBaseAsyncResult {
        public:
            using TFuture = NThreading::TFuture<T>;
            using TPromise = NThreading::TPromise<T>;

            virtual ~TBaseAsyncResult() = default;

            inline EResultCode ResultCode() const noexcept {
                return ResultCode_;
            }

        protected:
            inline TBaseAsyncResult(EResultCode resultCode) noexcept
                : ResultCode_(resultCode)
            {
            }

        private:
            EResultCode ResultCode_;
        };

        template <class T>
        class TAsyncResult: public TBaseAsyncResult<TAsyncResult<T>> {
        public:
            static inline TAsyncResult<T> Error(EResultCode resultCode) noexcept {
                Y_VERIFY(resultCode != OK);
                return TAsyncResult(resultCode);
            }
            static inline TAsyncResult<T> Success(T&& result) noexcept {
                return TAsyncResult(std::move(result));
            }

            inline bool Empty() const noexcept {
                return Result_.Empty();
            }
            inline const T& Result() const noexcept {
                return Result_.GetRef();
            }

        private:
            inline TAsyncResult(EResultCode resultCode) noexcept
                : TBaseAsyncResult<TAsyncResult<T>>(resultCode)
            {
            }
            inline TAsyncResult(T&& result) noexcept
                : TBaseAsyncResult<TAsyncResult<T>>(OK)
                , Result_(std::move(result))
            {
            }

            TMaybe<T, NMaybe::TPolicyUndefinedFail> Result_;
        };

        template <>
        class TAsyncResult<void>: public TBaseAsyncResult<TAsyncResult<void>> {
        public:
            static inline TAsyncResult<void> Error(EResultCode resultCode) noexcept {
                Y_VERIFY(resultCode != OK);
                return TAsyncResult(resultCode);
            }
            static inline TAsyncResult<void> Success() noexcept {
                return TAsyncResult(OK);
            }

        protected:
            using TBaseAsyncResult::TBaseAsyncResult;
        };

        class TDataTuple: public std::tuple<TString, TStat> {
        public:
            using tuple::tuple;

            inline const TString& Data() const noexcept {
                return std::get<0>(*this);
            }
            inline const TStat& Stat() const noexcept {
                return std::get<1>(*this);
            }
        };

        using TChildrenResult = TAsyncResult<TVector<TString>>;
        using TChildrenDataResult = TAsyncResult<THashMap<TString, TDataTuple>>;
        using TDataResult = TAsyncResult<TDataTuple>;
        using TPathResult = TAsyncResult<TString>;
        using TStatResult = TAsyncResult<TStat>;
        using TVoidResult = TAsyncResult<void>;

        class IBatchVisitor {
        public:
            virtual ~IBatchVisitor() = default;

            virtual void CreateOperation(const TString& path, const TString& data,
                                         ECreateMode createMode, TBuffer& resultingPath) = 0;
            virtual void DeleteOperation(const TString& path, int version) = 0;
            virtual void SetOperation(const TString& path, const TString& data, int version, TStat& stat) = 0;
            virtual void CheckOperation(const TString& path, int version) = 0;
        };

        class IAsyncOperation {
        public:
            using TRef = THolder<IAsyncOperation>;
            using TVectorType = THolderVector<IAsyncOperation>;

            virtual ~IAsyncOperation() = default;

            virtual void Prepare(IBatchVisitor& visitor) const = 0;
        };

        class TCreateOperation: public IAsyncOperation {
        public:
            TCreateOperation(const TString& path, const TString& data, ECreateMode createMode = ECreateMode::CM_PERSISTENT)
                : Path(path)
                , Data(data)
                , CreateMode(createMode)
                , ResultingPath(1024)
            {
            }

            void Prepare(IBatchVisitor& visitor) const override {
                visitor.CreateOperation(Path, Data, CreateMode, ResultingPath);
            }

        public:
            const TString Path;
            const TString Data;
            const ECreateMode CreateMode;
            mutable TBuffer ResultingPath;
        };

        class TDeleteOperation: public IAsyncOperation {
        public:
            TDeleteOperation(const TString& path, int version = -1)
                : Path(path)
                , Version(version)
            {
            }

            void Prepare(IBatchVisitor& visitor) const override {
                visitor.DeleteOperation(Path, Version);
            }

        public:
            const TString Path;
            const int Version;
        };

        class TSetOperation: public IAsyncOperation {
        public:
            TSetOperation(const TString& path, const TString& data, int version = -1)
                : Path(path)
                , Data(data)
                , Version(version)
            {
            }

            void Prepare(IBatchVisitor& visitor) const override {
                visitor.SetOperation(Path, Data, Version, Stat);
            }

        public:
            const TString Path;
            const TString Data;
            const int Version;
            mutable TStat Stat;
        };

        class TCheckOperation: public IAsyncOperation {
        public:
            TCheckOperation(const TString& path, int version = -1)
                : Path(path)
                , Version(version)
            {
            }

            void Prepare(IBatchVisitor& visitor) const override {
                visitor.CheckOperation(Path, Version);
            }

        public:
            const TString Path;
            const int Version;
        };

        class TClient: public TNonCopyable {
            Y_DECLARE_SINGLETON_FRIEND()
        public:
            ~TClient();

            // base methods, one-to-one mapping for zookeeper library
            [[nodiscard]] TVoidResult::TFuture Mutate(const IAsyncOperation::TVectorType& operations) noexcept;
            [[nodiscard]] TPathResult::TFuture Create(const TString& path, const TString& data, ECreateMode createMode = ECreateMode::CM_PERSISTENT) noexcept;
            [[nodiscard]] TVoidResult::TFuture Delete(const TString& path, int version = -1) noexcept;
            [[nodiscard]] TStatResult::TFuture SetData(const TString& path, const TString& data, int version = -1) noexcept;
            [[nodiscard]] TDataResult::TFuture GetData(const TString& path, bool watch = false) noexcept;
            [[nodiscard]] TChildrenResult::TFuture GetChildren(const TString& path, bool watch = false) noexcept;
            [[nodiscard]] TStatResult::TFuture Exists(const TString& path, bool watch = false) noexcept;

            // watcher api
            void AddWatcher(const TString& path, IWatcher* watcher);
            void RemoveWatcher(const TString& path, IWatcher* watcher);

            class TWatcherGuard: public TNonCopyable {
            public:
                TWatcherGuard(TClient& client, const TString& path, IWatcher* watcher) noexcept
                    : Client(client)
                    , Path(path)
                    , Watcher(watcher)
                {
                    if (Watcher) {
                        Client.AddWatcher(Path, Watcher);
                    }
                }

                TWatcherGuard(TWatcherGuard&& other) noexcept
                    : Client(other.Client)
                    , Path(other.Path)
                    , Watcher(other.Watcher)
                {
                    other.Watcher = nullptr;
                }

                ~TWatcherGuard() {
                    if (Watcher) {
                        Client.RemoveWatcher(Path, Watcher);
                    }
                }

            private:
                TClient& Client;
                const TString Path;
                IWatcher* Watcher;
            };

            [[nodiscard]] inline TWatcherGuard CreateWatcher(const TString& path, IWatcher* watcher) {
                return {*this, path, watcher};
            }

            // some useful recipes
            [[nodiscard]] TVoidResult::TFuture CreateRecursive(const TString& path) noexcept;
            [[nodiscard]] TChildrenDataResult::TFuture GetChildrenData(const TString& path, bool watchRoot = false, bool watchChildren = false) noexcept;

            static TClient* Get() {
                return SingletonWithPriority<TClient, 100003>();
            }

        private:
            TClient();

            class TImpl;
            THolder<TImpl> Impl;
        };
    }
}
