#include <infra/netmon/link_poller_hosts.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/topology_updater.h>

namespace NNetmon {
    class TLinkPollerHostsUpdater::TImpl: public TScheduledTask {
    public:
        using THostSet      = TLinkPollerHostsUpdater::THostSet;
        using THostSetRef   = TLinkPollerHostsUpdater::THostSetRef;
        using TSwitchSet    = TLinkPollerHostsUpdater::TSwitchSet;
        using TSwitchSetRef = TLinkPollerHostsUpdater::TSwitchSetRef;

        TImpl(const TTopologyStorage& topologyStorage)
            : TScheduledTask(TSettings::Get()->GetProbeScheduleTtl())
            , TopologyStorage(topologyStorage)
            , Hosts(MakeAtomicShared<THostSet>())
            , Switches(MakeAtomicShared<TSwitchSet>())
            , NewHosts()
        {
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                TSimpleTimer timer;

                auto newHosts = MakeAtomicShared<THostSet>();
                NewHosts.Swap(newHosts);
                auto newSwitches = CalculateSwitches(*newHosts);

                {
                    auto hosts = Hosts.OwnWrite();
                    auto switches = Switches.OwnWrite();
                    *hosts = newHosts;
                    *switches = newSwitches;
                }

                INFO_LOG << "Updated link poller hosts in " << timer.Get()
                         << ": " << newHosts->size() << " hosts, "
                         << newSwitches->size() << " switches" << Endl;
            });
        }

        void AddHost(const TTopology::THostRef& host) {
            NewHosts.Own()->insert(host);
        }

        THostSetRef GetHosts() const {
            return *Hosts.OwnRead();
        }

        TSwitchSetRef GetSwitches() const {
            return *Switches.OwnRead();
        }

    private:
        TSwitchSetRef CalculateSwitches(const THostSet& hosts) {
            auto switches = MakeAtomicShared<THashSet<TTopology::TSwitchRef>>();
            auto topology = TopologyStorage.GetTopology();
            topology->ForEachSwitch([&switches, &hosts](const TSwitch& switch_) {
                for (const auto host : switch_.GetRealHosts()) {
                    if (!hosts.contains(*host)) {
                        return;
                    }
                }
                switches->insert(TTopology::TSwitchRef(switch_));
            });
            return switches;
        }

        const TTopologyStorage& TopologyStorage;
        TRWLockedBox<THostSetRef> Hosts;
        TRWLockedBox<TSwitchSetRef> Switches;
        TAtomicLockedBox<THostSet> NewHosts;
    };

    TLinkPollerHostsUpdater::TLinkPollerHostsUpdater(const TTopologyStorage& topologyStorage)
        : Impl(MakeHolder<TImpl>(topologyStorage))
        , SchedulerGuard(Impl->Schedule())
    {
    }

    TLinkPollerHostsUpdater::~TLinkPollerHostsUpdater() = default;

    void TLinkPollerHostsUpdater::AddHost(const TTopology::THostRef& host) {
        Impl->AddHost(host);
    }

    TLinkPollerHostsUpdater::THostSetRef TLinkPollerHostsUpdater::GetHosts() const {
        return Impl->GetHosts();
    }

    TLinkPollerHostsUpdater::TSwitchSetRef TLinkPollerHostsUpdater::GetSwitches() const {
        return Impl->GetSwitches();
    }

    TThreadPool::TFuture TLinkPollerHostsUpdater::SpinAndWait() noexcept {
        return Impl->SpinAndWait();
    }
}
