#pragma once

#include <infra/netmon/library/metrics.h>
#include <infra/netmon/topology/types.h>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>
#include <library/cpp/unistat/unistat.h>
#include <library/cpp/unistat/raii.h>

#include <array>

namespace NNetmon {
    enum class ENetmonSignals {
        StateDumperDcWriteTime,
        StateDumperQueueWriteTime,
        StateDumperSwitchWriteTime,

        StateDumperDcReadTime,
        StateDumperQueueReadTime,
        StateDumperSwitchReadTime,

        SeriesDumperDcUpdateTime,
        SeriesDumperQueueUpdateTime,
        SeriesDumperSwitchUpdateTime,

        SeriesDumperDcWriteTime,
        SeriesDumperQueueWriteTime,
        SeriesDumperSwitchWriteTime,

        SeriesDumperDcReadTime,
        SeriesDumperQueueReadTime,
        SeriesDumperSwitchReadTime,

        PreAggregationDcTime,
        PreAggregationQueueTime,
        PreAggregationSwitchTime,

        MergeTime,

        ProbeSuccessCount,
        ProbeFailedCount,
        ProbeSemiFailedCount,

        ProbeReadingTime,
        ProbeInsertionTime,
        ProbeIncomingDelay,

        ProbeSchedulingTime,

        HistoryInsertionTime,

        SlicerDcCollectTime,
        SlicerQueueCollectTime,
        SlicerSwitchCollectTime,
        SlicerStateReplyTime,

        StateDcAge,
        StateQueueAge,
        StateSwitchAge,

        ApiNewClientReports,
        ApiDroppedClientReports,
        ApiClientReportsCount,

        InterestedHostsCount,

        SliceDcIndexTotal,
        SliceLineIndexTotal,
        SliceSwitchIndexTotal,

        MemPoolProbes,
        MemPoolBatches,
        MemPoolPairs,
        MemPoolBlocks,

        HttpRequests,
        HttpServerFailedRequests,
        HttpServerMaxConnErrors,
        HttpServerUncaughtExceptions,
   };

    inline TString ToString(const ENetmonSignals v) {
        switch (v) {
            case ENetmonSignals::StateDumperDcWriteTime:
                return "state_dumper_dc_write_time";
            case ENetmonSignals::StateDumperQueueWriteTime:
                return "state_dumper_queue_write_time";
            case ENetmonSignals::StateDumperSwitchWriteTime:
                return "state_dumper_switch_write_time";
            case ENetmonSignals::StateDumperDcReadTime:
                return "state_dumper_dc_read_time";
            case ENetmonSignals::StateDumperQueueReadTime:
                return "state_dumper_queue_read_time";
            case ENetmonSignals::StateDumperSwitchReadTime:
                return "state_dumper_switch_read_time";
            case ENetmonSignals::SeriesDumperDcUpdateTime:
                return "series_dumper_dc_update_time";
            case ENetmonSignals::SeriesDumperQueueUpdateTime:
                return "series_dumper_queue_update_time";
            case ENetmonSignals::SeriesDumperSwitchUpdateTime:
                return "series_dumper_switch_update_time";
            case ENetmonSignals::SeriesDumperDcWriteTime:
                return "series_dumper_dc_write_time";
            case ENetmonSignals::SeriesDumperQueueWriteTime:
                return "series_dumper_queue_write_time";
            case ENetmonSignals::SeriesDumperSwitchWriteTime:
                return "series_dumper_switch_write_time";
            case ENetmonSignals::SeriesDumperDcReadTime:
                return "series_dumper_dc_read_time";
            case ENetmonSignals::SeriesDumperQueueReadTime:
                return "series_dumper_queue_read_time";
            case ENetmonSignals::SeriesDumperSwitchReadTime:
                return "series_dumper_switch_read_time";
            case ENetmonSignals::PreAggregationDcTime:
                return "pre_aggregation_dc_time";
            case ENetmonSignals::PreAggregationQueueTime:
                return "pre_aggregation_queue_time";
            case ENetmonSignals::PreAggregationSwitchTime:
                return "pre_aggregation_switch_time";
            case ENetmonSignals::MergeTime:
                return "merge_time";
            case ENetmonSignals::ProbeSuccessCount:
                return "probe_success_count";
            case ENetmonSignals::ProbeFailedCount:
                return "probe_failed_count";
            case ENetmonSignals::ProbeSemiFailedCount:
                return "probe_semifailed_count";
            case ENetmonSignals::ProbeReadingTime:
                return "probe_reading_time";
            case ENetmonSignals::ProbeInsertionTime:
                return "probe_insertion_time";
            case ENetmonSignals::ProbeIncomingDelay:
                return "probe_incoming_delay";
            case ENetmonSignals::ProbeSchedulingTime:
                return "probe_scheduling_time";
            case ENetmonSignals::HistoryInsertionTime:
                return "history_insertion_time";
            case ENetmonSignals::SlicerDcCollectTime:
                return "slice_collector_dc_time";
            case ENetmonSignals::SlicerQueueCollectTime:
                return "slice_collector_queue_time";
            case ENetmonSignals::SlicerSwitchCollectTime:
                return "slice_collector_switch_time";
            case ENetmonSignals::SlicerStateReplyTime:
                return "slicer_state_reply_time";
            case ENetmonSignals::StateDcAge:
                return "state_dc_age";
            case ENetmonSignals::StateQueueAge:
                return "state_queue_age";
            case ENetmonSignals::StateSwitchAge:
                return "state_switch_age";
            case ENetmonSignals::ApiNewClientReports:
                return "api_new_client_reports";
            case ENetmonSignals::ApiDroppedClientReports:
                return "api_dropped_client_reports";
            case ENetmonSignals::ApiClientReportsCount:
                return "api_client_reports_count";
            case ENetmonSignals::InterestedHostsCount:
                return "interested_hosts_count";
            case ENetmonSignals::SliceDcIndexTotal:
                return "dc_index_total_count";
            case ENetmonSignals::SliceLineIndexTotal:
                return "line_index_total_count";
            case ENetmonSignals::SliceSwitchIndexTotal:
                return "switch_index_total_count";
            case ENetmonSignals::MemPoolProbes:
                return "mem_pool_probes_count";
            case ENetmonSignals::MemPoolBatches:
                return "mem_pool_batches_count";
            case ENetmonSignals::MemPoolPairs:
                return "mem_pool_pairs_count";
            case ENetmonSignals::MemPoolBlocks:
                return "mem_pool_blocks_count";

            case ENetmonSignals::HttpRequests:
                return "http_requests";
            case ENetmonSignals::HttpServerFailedRequests:
                return "http_failed_requests";
            case ENetmonSignals::HttpServerMaxConnErrors:
                return "http_max_conn_errors";
            case ENetmonSignals::HttpServerUncaughtExceptions:
                return "http_uncaught_exceptions";
       };
    }

    class TStatsInitializer {
    public:
        void Init(TUnistat& t) const {
            auto priority = NUnistat::TPriority(10);

            t.DrillFloatHole("probe_success_count", "summ", priority);
            t.DrillFloatHole("probe_failed_count", "summ", priority);
            t.DrillFloatHole("probe_semifailed_count", "summ", priority);

            t.DrillFloatHole("api_new_client_reports", "summ", priority);
            t.DrillFloatHole("api_dropped_client_reports", "summ", priority);
            const NUnistat::TIntervals reportsCountBuckets = {
                1, 10, 50, 100, 200, 300, 400, 500, 600, 700, 800, 900, 1000
            };
            t.DrillHistogramHole("api_client_reports_count", "hgram", priority, reportsCountBuckets);

            t.DrillFloatHole("interested_hosts_count", "ammv", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);

            t.DrillFloatHole("dc_index_total_count", "ammv", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);
            t.DrillFloatHole("line_index_total_count", "ammv", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);
            t.DrillFloatHole("switch_index_total_count", "ammv", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);

            t.DrillFloatHole("mem_pool_probes_count", "ammv", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);
            t.DrillFloatHole("mem_pool_batches_count", "ammv", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);
            t.DrillFloatHole("mem_pool_pairs_count", "ammv", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);
            t.DrillFloatHole("mem_pool_blocks_count", "ammv", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);

            const auto& intervals = GetYasmBuckets();

            t.DrillHistogramHole("state_dumper_dc_read_time", "hgram", priority, intervals);
            t.DrillHistogramHole("state_dumper_queue_read_time", "hgram", priority, intervals);
            t.DrillHistogramHole("state_dumper_switch_read_time", "hgram", priority, intervals);

            t.DrillHistogramHole("state_dumper_dc_write_time", "hgram", priority, intervals);
            t.DrillHistogramHole("state_dumper_queue_write_time", "hgram", priority, intervals);
            t.DrillHistogramHole("state_dumper_switch_write_time", "hgram", priority, intervals);

            t.DrillHistogramHole("series_dumper_dc_update_time", "hgram", priority, intervals);
            t.DrillHistogramHole("series_dumper_queue_update_time", "hgram", priority, intervals);
            t.DrillHistogramHole("series_dumper_switch_update_time", "hgram", priority, intervals);

            t.DrillHistogramHole("series_dumper_dc_write_time", "hgram", priority, intervals);
            t.DrillHistogramHole("series_dumper_queue_write_time", "hgram", priority, intervals);
            t.DrillHistogramHole("series_dumper_switch_write_time", "hgram", priority, intervals);

            t.DrillHistogramHole("series_dumper_dc_read_time", "hgram", priority, intervals);
            t.DrillHistogramHole("series_dumper_queue_read_time", "hgram", priority, intervals);
            t.DrillHistogramHole("series_dumper_switch_read_time", "hgram", priority, intervals);

            t.DrillHistogramHole("pre_aggregation_dc_time", "hgram", priority, intervals);
            t.DrillHistogramHole("pre_aggregation_queue_time", "hgram", priority, intervals);
            t.DrillHistogramHole("pre_aggregation_switch_time", "hgram", priority, intervals);

            t.DrillHistogramHole("merge_time", "hgram", priority, intervals);

            t.DrillHistogramHole("probe_reading_time", "hgram", priority, intervals);
            t.DrillHistogramHole("probe_insertion_time", "hgram", priority, intervals);
            t.DrillHistogramHole("probe_incoming_delay", "hgram", priority, intervals);

            t.DrillHistogramHole("probe_scheduling_time", "hgram", priority, intervals);

            t.DrillHistogramHole("history_insertion_time", "hgram", priority, intervals);

            t.DrillHistogramHole("slice_collector_dc_time", "hgram", priority, intervals);
            t.DrillHistogramHole("slice_collector_queue_time", "hgram", priority, intervals);
            t.DrillHistogramHole("slice_collector_switch_time", "hgram", priority, intervals);

            t.DrillHistogramHole("slicer_state_reply_time", "hgram", priority, intervals);

            t.DrillHistogramHole("state_dc_age", "hgram", priority, intervals);
            t.DrillHistogramHole("state_queue_age", "hgram", priority, intervals);
            t.DrillHistogramHole("state_switch_age", "hgram", priority, intervals);

            t.DrillFloatHole("http_requests", "summ", priority, NUnistat::TStartValue(0), EAggregationType::LastValue);
            t.DrillFloatHole("http_failed_requests", "summ", priority);
            t.DrillFloatHole("http_max_conn_errors", "summ", priority);
            t.DrillFloatHole("http_uncaught_exceptions", "summ", priority);
       }
    };

    enum class EProbeScore {
        INVALID,
        SUCCESS,
        FAILED,
        SEMIFAILED
    };

    static inline EProbeScore GetScoreClass(double score) {
        if (score == 1.0) {
            return EProbeScore::SUCCESS;
        } else if (score == 0) {
            return EProbeScore::FAILED;
        } else if (score > 0) {
            return EProbeScore::SEMIFAILED;
        } else {
            return EProbeScore::INVALID;
        }
    }

    template <class T>
    struct TGenericPacketCounters {
        using TValue = T;

        template <class U>
        using TGenericType = TGenericPacketCounters<U>;

        static TGenericPacketCounters<ui64> ExtractMetrics(TGenericPacketCounters<TAtomic>& counters) {
            return TGenericPacketCounters<ui64>{
                static_cast<TValue>(AtomicSwap(&counters.Success, 0)),
                static_cast<TValue>(AtomicSwap(&counters.Failed, 0)),
                static_cast<TValue>(AtomicSwap(&counters.TosChanged, 0)),
            };
        }

        TValue Success = 0, Failed = 0, TosChanged = 0;
    };

    static inline void IncrementCounter(TGenericPacketCounters<TAtomic>& counters, ui64 success, ui64 failed, ui64 changed) {
        AtomicAdd(counters.Success, success);
        AtomicAdd(counters.Failed, failed);
        AtomicAdd(counters.TosChanged, changed);
    }

    static inline TGenericPacketCounters<ui64>& operator+=(TGenericPacketCounters<ui64>& lhs, const TGenericPacketCounters<ui64>& rhs) {
        lhs.Success += rhs.Success;
        lhs.Failed  += rhs.Failed;
        lhs.TosChanged  += rhs.TosChanged;
        return lhs;
    }

    static inline TGenericPacketCounters<ui64>& operator-=(TGenericPacketCounters<ui64>& lhs, const TGenericPacketCounters<ui64>& rhs) {
        lhs.Success -= rhs.Success;
        lhs.Failed  -= rhs.Failed;
        lhs.TosChanged  -= rhs.TosChanged;
        return lhs;
    }

    static inline bool IsEmptyMetric(const TGenericPacketCounters<ui64>& counters) {
        return counters.Success == 0 && counters.Failed == 0;
    }

    static inline TMaybe<double> CalculateNocSlaMetric(const TGenericPacketCounters<ui64>& counters) {
        auto sum = counters.Success + counters.Failed;
        if (sum > 0) {
            return static_cast<double>(counters.Success) / sum;
        } else {
            return Nothing();
        }
    }

    static inline TMaybe<double> CalculateNocSlaTosMetric(const TGenericPacketCounters<ui64>& counters) {
        if (counters.Success > 0) {
            return static_cast<double>(counters.Success - counters.TosChanged) / counters.Success;
        } else {
            return Nothing();
        }
    }

    template <class T>
    struct TGenericRttCounters;

    template <class T>
    struct TGenericRttCountersBase {
        using TValue = T;

        template <class U>
        using TGenericType = TGenericRttCounters<U>;

        static const size_t RTT_BUCKET_COUNT = 64;
        using TBuckets = std::array<TValue, RTT_BUCKET_COUNT>;

        TGenericRttCountersBase()
            : Buckets(MakeAtomicShared<TBuckets>())
        {
        }

        TAtomicSharedPtr<TBuckets> Buckets;
    };

    template <>
    struct TGenericRttCounters<TAtomic> : public TGenericRttCountersBase<TAtomic> {
        TGenericRttCounters() = default;

        TLightRWLock Lock;
    };

    void IncrementCounter(TGenericRttCounters<TAtomic>& counters, double rtt);

    template <>
    struct TGenericRttCounters<ui64> : public TGenericRttCountersBase<ui64> {
        static TGenericType<ui64> ExtractMetrics(TGenericType<TAtomic>& counters) {
            auto buckets = MakeAtomicShared<TGenericType<TAtomic>::TBuckets>();
            {
                TLightWriteGuard guard(counters.Lock);
                counters.Buckets.Swap(buckets);
            }
            TGenericType<ui64> ret;
            Copy(begin(*buckets), end(*buckets), begin(*ret.Buckets));
            return ret;
        }

    };

    static inline bool IsEmptyMetric(const TGenericRttCounters<ui64>& counters) {
        return AllOf(*counters.Buckets, [](ui64 value) { return value == 0; });
    }

    static inline TGenericRttCounters<ui64>& operator+=(TGenericRttCounters<ui64>& lhs, const TGenericRttCounters<ui64>& rhs) {
        for (size_t i = 0; i < TGenericRttCounters<ui64>::RTT_BUCKET_COUNT; ++i) {
            (*lhs.Buckets)[i] += (*rhs.Buckets)[i];
        }
        return lhs;
    }

    static inline TGenericRttCounters<ui64>& operator-=(TGenericRttCounters<ui64>& lhs, const TGenericRttCounters<ui64>& rhs) {
        for (size_t i = 0; i < TGenericRttCounters<ui64>::RTT_BUCKET_COUNT; ++i) {
            (*lhs.Buckets)[i] -= (*rhs.Buckets)[i];
        }
        return lhs;
    }

    namespace NInternal {
        // unified code for both rtt and packet counters
        template <class TBaseCounters>
        struct TTrafficClassCounters {
            using TValue = typename TBaseCounters::TValue;
            template <class T> using TGenericType =
                TTrafficClassCounters<typename TBaseCounters::template TGenericType<T>>;
            using TAtomicBaseCounters = typename TBaseCounters::template TGenericType<TAtomic>;

            static TGenericType<ui64> ExtractMetrics(TGenericType<TAtomic>& counters) {
                return TGenericType<ui64>{
                    TBaseCounters::ExtractMetrics(counters.CS0),
                    TBaseCounters::ExtractMetrics(counters.CS1),
                    TBaseCounters::ExtractMetrics(counters.CS2),
                    TBaseCounters::ExtractMetrics(counters.CS3),
                    TBaseCounters::ExtractMetrics(counters.CS4),
                    TBaseCounters::ExtractMetrics(counters.BB4),
                    TBaseCounters::ExtractMetrics(counters.SparseBB),
                    TBaseCounters::ExtractMetrics(counters.SparseFB)
                };
            }

            // full-mesh probes, both dom0 and mtn vlans
            TBaseCounters CS0, CS1, CS2, CS3, CS4;
            TBaseCounters BB4;
            // sparse probes, only dom0 vlans inside RTC
            TBaseCounters SparseBB, SparseFB;
        };

        template <class TBaseCounters,
                  class = typename std::enable_if_t<std::is_same_v<typename TBaseCounters::TValue, ui64>>>
        inline TTrafficClassCounters<TBaseCounters>& operator+=(TTrafficClassCounters<TBaseCounters>& lhs, const TTrafficClassCounters<TBaseCounters>& rhs) {
            lhs.CS0 += rhs.CS0;
            lhs.CS1 += rhs.CS1;
            lhs.CS2 += rhs.CS2;
            lhs.CS3 += rhs.CS3;
            lhs.CS4 += rhs.CS4;
            lhs.BB4 += rhs.BB4;
            lhs.SparseBB += rhs.SparseBB;
            lhs.SparseFB += rhs.SparseFB;
            return lhs;
        }

        template <class TBaseCounters,
                  class = typename std::enable_if_t<std::is_same_v<typename TBaseCounters::TValue, ui64>>>
        inline TTrafficClassCounters<TBaseCounters>& operator-=(TTrafficClassCounters<TBaseCounters>& lhs, const TTrafficClassCounters<TBaseCounters>& rhs) {
            lhs.CS0 -= rhs.CS0;
            lhs.CS1 -= rhs.CS1;
            lhs.CS2 -= rhs.CS2;
            lhs.CS3 -= rhs.CS3;
            lhs.CS4 -= rhs.CS4;
            lhs.BB4 -= rhs.BB4;
            lhs.SparseBB -= rhs.SparseBB;
            lhs.SparseFB -= rhs.SparseFB;
            return lhs;
        }

        template <class T, class... Args>
        inline void IncrementCounter(TTrafficClassCounters<T>& counters,
                                     ENetworkType network,
                                     Args&&... args) {
            static_assert(std::is_same_v<typename T::TValue, TAtomic>);
            switch (network) {
                case ENetworkType::FASTBONE6:
                    // All fastbone probes are colored, since agents currently
                    // differentiate bb & fb only by color.
                    break;
                case ENetworkType::FASTBONE_CS1:
                    IncrementCounter(counters.CS1, std::forward<Args>(args)...);
                    break;
                case ENetworkType::FASTBONE_CS2:
                    IncrementCounter(counters.CS2, std::forward<Args>(args)...);
                    break;
                case ENetworkType::BACKBONE6:
                    IncrementCounter(counters.CS0, std::forward<Args>(args)...);
                    [[fallthrough]]; // TODO: put `break;` here when we add proper CS3 probes
                case ENetworkType::BACKBONE_CS3:
                    // CS3 is the default color for backbone
                    IncrementCounter(counters.CS3, std::forward<Args>(args)...);
                    break;
                case ENetworkType::BACKBONE_CS4:
                    IncrementCounter(counters.CS4, std::forward<Args>(args)...);
                    break;
                case ENetworkType::BACKBONE4:
                    IncrementCounter(counters.BB4, std::forward<Args>(args)...);
                    break;
                case ENetworkType::SPARSE_BACKBONE6:
                    IncrementCounter(counters.SparseBB, std::forward<Args>(args)...);
                    break;
                case ENetworkType::SPARSE_FASTBONE6:
                    IncrementCounter(counters.SparseFB, std::forward<Args>(args)...);
                    break;
                case ENetworkType::NIL_NETWORK:
                    break;
            }
        }

        template <class TBaseCounters>
        static inline bool IsEmptyMetric(const TTrafficClassCounters<TBaseCounters>& counters) {
            return IsEmptyMetric(counters.CS0) &&
                   IsEmptyMetric(counters.CS1) &&
                   IsEmptyMetric(counters.CS2) &&
                   IsEmptyMetric(counters.CS3) &&
                   IsEmptyMetric(counters.CS4) &&
                   IsEmptyMetric(counters.BB4) &&
                   IsEmptyMetric(counters.SparseBB) &&
                   IsEmptyMetric(counters.SparseFB);
        }
    }

    template <class TValue>
    using TGenericPacketSlaCounters = NInternal::TTrafficClassCounters<TGenericPacketCounters<TValue>>;
    template <class TValue>
    using TGenericRttSlaCounters = NInternal::TTrafficClassCounters<TGenericRttCounters<TValue>>;

    using TPacketSlaCounters = TGenericPacketSlaCounters<TAtomic>;
    using TRttSlaCounters = TGenericRttSlaCounters<TAtomic>;

    class TDatacenter;
    class TTopologyStorage;

    class TCrossDcCounters {
    public:
        TCrossDcCounters(const TTopologyStorage& topologyStorage);

        void RegisterPackets(const TDatacenter& sourceDc, const TDatacenter& targetDc,
                             ENetworkType network, ui64 successCount, ui64 failCount,
                             ui64 changedCount, double rtt);

        TVector<THashMap<ui64, TPacketSlaCounters>>& GetPacketCounterMapList() {
            return PacketCountersList;
        }
        TVector<THashMap<ui64, TRttSlaCounters>>& GetRttCounterMapList() {
            return RttCountersList;
        }

    private:
        TVector<THashMap<ui64, TPacketSlaCounters>> PacketCountersList;
        TVector<THashMap<ui64, TRttSlaCounters>> RttCountersList;

        THashMap<ui64, size_t> DcIndexMapping;
    };
}
