#include <infra/netmon/metrics_updater.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/library/helpers.h>
#include <infra/netmon/library/requester.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/statistics/histograms.h>
#include <infra/netmon/topology/topology_storage.h>

#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/algorithm.h>
#include <util/generic/is_in.h>
#include <util/generic/queue.h>
#include <util/generic/xrange.h>
#include <util/system/hostname.h>

namespace NNetmon {
    namespace {
        const TDuration YASM_DEFAULT_INTERVAL = TDuration::Seconds(5);
        const TDuration SOLOMON_PUSH_INTERVAL = TDuration::Seconds(15);

        const TString MTN_BACKBONE_VLAN_LABEL = ::ToString(static_cast<int>(EVlans::MTN_BACKBONE));
        const TString MTN_FASTBONE_VLAN_LABEL = ::ToString(static_cast<int>(EVlans::MTN_FASTBONE));
        const TString DOM0_BACKBONE_VLAN_LABEL = "bb_default";
        const TString DOM0_FASTBONE_VLAN_LABEL = "fb_default";

        template <template<class> class TGenericCounters>
        class TMultiCounterSlidingWindow {
        public:
            using TMetrics = TGenericCounters<ui64>;
            using TCounters = TGenericCounters<TAtomic>;

            TMultiCounterSlidingWindow(TDuration window)
                : MetricsCount(window / YASM_DEFAULT_INTERVAL + 1)
            {
            }

            TMetrics Shift(TCounters* counters) {
                if (Metrics.size() == MetricsCount) {
                    Accumulator -= Metrics.front();
                    Metrics.pop();
                }

                if (counters) {
                    Metrics.emplace(TMetrics::ExtractMetrics(*counters)); // counters get cleared here
                } else {
                    Metrics.emplace();
                }
                Accumulator += Metrics.back();

                return Accumulator;
            }

        private:
            TQueue<TMetrics> Metrics;
            size_t MetricsCount;
            TMetrics Accumulator;
        };

        template <template<class> class TGenericCounters>
        class TSlidingWindow {
        public:
            using TMetrics = TGenericCounters<ui64>;
            using TCounters = TGenericCounters<TAtomic>;

            TSlidingWindow(TDuration window,
                           TCounters& counters)
                : Internal(window)
                , Counters(counters)
            {
            }

            TMetrics Shift() {
                return Internal.Shift(&Counters);
            }

        private:
            TMultiCounterSlidingWindow<TGenericCounters> Internal;
            TCounters& Counters;
        };

        template <class TConnectivityMetrics>
        void AddConnectivityMetrics(TVector<TSolomonPusher::TMetric>& pushMetrics,
                                    const TConnectivityMetrics& metrics,
                                    const TString& sensor,
                                    const TString& dc,
                                    std::initializer_list<NMonitoring::TLabel> extraLabels,
                                    bool downtime) {
            if (auto value = CalculateNocSlaMetric(metrics)) {
                TString rawSensor = sensor + "_raw";
                pushMetrics.emplace_back(TSolomonPusher::MakeMetric(*value, rawSensor, dc, extraLabels));

                if (!downtime) {
                    pushMetrics.emplace_back(TSolomonPusher::MakeMetric(*value, sensor, dc, extraLabels));
                }
            }

            if (auto value = CalculateNocSlaTosMetric(metrics)) {
                TString tosSensor = sensor + "_tos";
                TString rawTosSensor = tosSensor + "_raw";
                pushMetrics.emplace_back(TSolomonPusher::MakeMetric(*value, rawTosSensor, dc, extraLabels));

                if (!downtime) {
                    pushMetrics.emplace_back(TSolomonPusher::MakeMetric(*value, tosSensor, dc, extraLabels));
                }
            }
        }

        void AddRttPercentileMetrics(TVector<TSolomonPusher::TMetric>& pushMetrics,
                                     double value,
                                     const TString& sensor,
                                     const TString& percentile,
                                     const TString& dc,
                                     std::initializer_list<NMonitoring::TLabel> extraLabels,
                                     bool downtime) {
            TString rawSensor = sensor + "_raw";
            pushMetrics.emplace_back(TSolomonPusher::MakePercentileMetric(
                value, rawSensor, percentile, dc, extraLabels
            ));
            if (!downtime) {
                pushMetrics.emplace_back(TSolomonPusher::MakePercentileMetric(
                    value, sensor, percentile, dc, extraLabels
                ));
            }
        }

        void AddRttMetrics(TVector<TSolomonPusher::TMetric>& pushMetrics,
                           const TGenericRttCounters<ui64>& metrics,
                           const TString& sensor,
                           const TString& dc,
                           std::initializer_list<NMonitoring::TLabel> extraLabels,
                           bool downtime) {
            TSampleHistogram hist(*metrics.Buckets);
            TSampleHistogram::TBuckets cumulativeCount;
            hist.FillCumulativeCount(cumulativeCount);

            if (auto p50 = hist.GetPercentileValue(0.50, cumulativeCount)) {
                AddRttPercentileMetrics(pushMetrics, *p50, sensor, "50", dc, extraLabels, downtime);
            }
            if (auto p90 = hist.GetPercentileValue(0.90, cumulativeCount)) {
                AddRttPercentileMetrics(pushMetrics, *p90, sensor, "90", dc, extraLabels, downtime);
            }
            if (auto p95 = hist.GetPercentileValue(0.95, cumulativeCount)) {
                AddRttPercentileMetrics(pushMetrics, *p95, sensor, "95", dc, extraLabels, downtime);
            }
            if (auto p99 = hist.GetPercentileValue(0.99, cumulativeCount)) {
                AddRttPercentileMetrics(pushMetrics, *p99, sensor, "99", dc, extraLabels, downtime);
            }
        }
    }

    extern TVector<TPacketSlaCounters> PacketCounters;
    extern TVector<TRttSlaCounters> RttCounters;

    class TSolomonPusher::TImpl: public TScheduledTask {
    public:
        static constexpr size_t FAILED_ATTEMPTS_THRESHOLD = 10;

        TImpl(const TString& pushUrl, const TIamUpdater& iamUpdater)
            : TScheduledTask(pushUrl ? SOLOMON_PUSH_INTERVAL : TDuration())
            , PushUrl(pushUrl)
            , IamUpdater(iamUpdater)
            , PushBuilderBox()
        {
        }

        TThreadPool::TFuture Run() override {
            return Push();
        }

        template <class TMetricsContainer>
        void RecordMetrics(const TMetricsContainer& metrics) {
            PushBuilderBox.Own()->RecordMetrics(metrics);
        }

    private:
        class TPushBuilder {
        public:
            TPushBuilder()
                : Output(Data)
                , MetricRegistry(GetCommonLabels())
                , Encoder(NMonitoring::BufferedEncoderJson(&Output))
            {
            }

            template <class TMetricsContainer>
            void RecordMetrics(const TMetricsContainer& metrics) {
                for (auto& metric : metrics) {
                    auto* gauge = MetricRegistry.Gauge(metric.second);
                    gauge->Set(metric.first);
                }
                auto timestamp = RoundInstant(TInstant::Now(), TDuration::Seconds(5));
                MetricRegistry.Accept(timestamp, Encoder.Get());
            }

            TString Finish() {
                Encoder->Close();
                return std::move(Data);
            }

        private:
            static NMonitoring::TLabels GetCommonLabels() {
                NMonitoring::TLabels labels({{"host", HostName()}});
                for (const auto& [k, v] : TSettings::Get()->GetSolomonLabels()) {
                    labels.Add(k, v);
                }
                return labels;
            }

            TString Data;
            TStringOutput Output;
            NMonitoring::TMetricRegistry MetricRegistry;
            NMonitoring::IMetricEncoderPtr Encoder;
        };

        NThreading::TFuture<void> Push() {
            if (!FinishedPushData.Defined() || FinishedPushData->second > FAILED_ATTEMPTS_THRESHOLD) {
                auto pushBuilder = MakeAtomicShared<TPushBuilder>();
                PushBuilderBox.Swap(pushBuilder);

                if (FinishedPushData.Defined()) {
                    ERROR_LOG << "Solomon pusher reached " << FAILED_ATTEMPTS_THRESHOLD
                              << " failed attempts, dropping data" << Endl;
                }
                FinishedPushData.ConstructInPlace(pushBuilder->Finish(), 0);
            }

            TString authHeader;
            if (TSettings::Get()->GetSolomonPushToken()) {
                authHeader = "Authorization: OAuth " + TSettings::Get()->GetSolomonPushToken();
            } else {
                authHeader = "Authorization: Bearer " + IamUpdater.GetToken();
            }

            auto promise = NThreading::NewPromise<void>();
            THttpRequester::Get()->MakeRequest(
                PushUrl,
                {authHeader},
                NHttp::TFetchOptions()
                    .SetContentType(TString(NMonitoring::NFormatContenType::JSON))
                    .SetPostData(FinishedPushData->first)
            ).Subscribe([promise, this](const THttpRequester::TFuture& future) mutable {
                try {
                    future.GetValue();
                    FinishedPushData.Clear();
                } catch(...) {
                    WARNING_LOG << "Push to Solomon failed: " << CurrentExceptionMessage() << Endl;
                    ++FinishedPushData->second;
                }
                promise.SetValue();
            });
            return promise;
        }

        const TString PushUrl;
        const TIamUpdater& IamUpdater;
        TAtomicLockedBox<TPushBuilder> PushBuilderBox;
        TMaybe<std::pair<TString, size_t>> FinishedPushData;
        // FinishedPushData is null if last push was successful;
        // otherwise it stores the data we failed to push and the number of failed push attempts.
    };

    TSolomonPusher::TSolomonPusher(const TString& pushUrl, const TIamUpdater& iamUpdater)
        : Impl(MakeHolder<TImpl>(pushUrl, iamUpdater))
        , SchedulerGuard(pushUrl ? Impl->Schedule() : nullptr)
    {
    }

    TSolomonPusher::~TSolomonPusher() = default;

    template <class TMetricsContainer>
    void TSolomonPusher::RecordMetrics(const TMetricsContainer& metrics) {
        Impl->RecordMetrics(metrics);
    }

    TCommonMetricsUpdater::TCommonMetricsUpdater()
        : TBaseMetricsUpdater(YASM_DEFAULT_INTERVAL)
        , SchedulerGuard(Schedule())
    {
    }

    void TCommonMetricsUpdater::Update() {
        extern TAtomic MemPoolStatsProbeCount;
        extern TAtomic MemPoolStatsPairStateCount;
        extern TAtomic MemPoolStatsFixedBlockCount;
        extern TAtomic MemPoolStatsBatchCount;

        extern TAtomic SliceStatsDcIndexTotal;
        extern TAtomic SliceStatsLineIndexTotal;
        extern TAtomic SliceStatsSwitchIndexTotal;

        extern TAtomic InterestedHostsCount;
        extern TAtomic HttpRequestCount;

        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::MemPoolProbes, AtomicGet(MemPoolStatsProbeCount));
        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::MemPoolBatches, AtomicGet(MemPoolStatsBatchCount));
        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::MemPoolPairs, AtomicGet(MemPoolStatsPairStateCount));
        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::MemPoolBlocks, AtomicGet(MemPoolStatsFixedBlockCount));

        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::SliceDcIndexTotal, SliceStatsDcIndexTotal);
        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::SliceLineIndexTotal, SliceStatsLineIndexTotal);
        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::SliceSwitchIndexTotal, SliceStatsSwitchIndexTotal);

        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::InterestedHostsCount, AtomicGet(InterestedHostsCount));
        TUnistat::Instance().PushSignalUnsafe(ENetmonSignals::HttpRequests, AtomicGet(HttpRequestCount));
    }

    class TSlaMetricsUpdater::TImpl: public TNonCopyable {
    public:
        TImpl(TDuration window, TSolomonPusher& solomonPusher, const TInfraUpdater& infraUpdater)
            : SolomonPusher(solomonPusher)
            , InfraUpdater(infraUpdater)
            , GracePeriodEnd((2 * window).ToDeadLine())
        {
            size_t size = TSettings::Get()->GetProbeScheduleDcs().size();
            PacketCounters.resize(size);
            TVector<TRttSlaCounters>(size).swap(RttCounters);

            for (auto& packetCounter : PacketCounters) {
                PacketWindows.emplace_back(TPacketWindow(window, packetCounter));
            }
            for (auto& rttCounter : RttCounters) {
                RttWindows.emplace_back(TRttWindow(window, rttCounter));
            }
        }

        void Update() {
            const auto& dcs = TSettings::Get()->GetProbeScheduleDcs();
            for (size_t i = 0; i < dcs.size(); ++i) {
                const auto& dc = dcs[i];
                bool downtime = InfraUpdater.IsInfraEventOngoing(dc);

                UpdateWindow(PacketWindows[i], dc, "nocsla", downtime);
                UpdateWindow(RttWindows[i], dc, downtime);
            }
        }

    private:
        using TPacketWindow = TSlidingWindow<TGenericPacketSlaCounters>;
        using TRttWindow    = TSlidingWindow<TGenericRttSlaCounters>;

        template <class T>
        void UpdateWindow(T& window, const TString& solomonDc, const TString& sensor, bool downtime) {
            auto metrics = window.Shift();
            if (TInstant::Now() <= GracePeriodEnd) {
                return;
            }

            PushToSolomon(metrics, solomonDc, sensor, downtime);
        }

        void UpdateWindow(TRttWindow& window, const TString& solomonDc, bool downtime) {
            auto metrics = window.Shift();
            if (TInstant::Now() <= GracePeriodEnd) {
                return;
            }

            TVector<TSolomonPusher::TMetric> solomonMetrics;
            AddRttMetrics(solomonMetrics, metrics.CS1, "rtt", solomonDc, {{"tc", "cs1"}}, downtime);
            AddRttMetrics(solomonMetrics, metrics.CS2, "rtt", solomonDc, {{"tc", "cs2"}}, downtime);
            AddRttMetrics(solomonMetrics, metrics.CS3, "rtt", solomonDc, {{"tc", "cs3"}}, downtime);
            AddRttMetrics(solomonMetrics, metrics.CS4, "rtt", solomonDc, {{"tc", "cs4"}}, downtime);
            AddRttMetrics(solomonMetrics, metrics.BB4, "rtt", solomonDc, {{"network", "bb4"}}, downtime);
            AddRttMetrics(solomonMetrics, metrics.SparseBB, "rtt", solomonDc, {{"schedule", "sparse"}, {"network", "bb6"}}, downtime);
            AddRttMetrics(solomonMetrics, metrics.SparseFB, "rtt", solomonDc, {{"schedule", "sparse"}, {"network", "fb6"}}, downtime);

            SolomonPusher.RecordMetrics(solomonMetrics);
        }

        template <class TMetrics>
        void PushToSolomon(const TMetrics& metrics, const TString& solomonDc, const TString& sensor, bool downtime) {
            TVector<TSolomonPusher::TMetric> solomonMetrics;
            AddConnectivityMetrics(solomonMetrics, metrics.CS1, sensor, solomonDc, {{"tc", "cs1"}}, downtime);
            AddConnectivityMetrics(solomonMetrics, metrics.CS2, sensor, solomonDc, {{"tc", "cs2"}}, downtime);
            AddConnectivityMetrics(solomonMetrics, metrics.CS3, sensor, solomonDc, {{"tc", "cs3"}}, downtime);
            AddConnectivityMetrics(solomonMetrics, metrics.CS4, sensor, solomonDc, {{"tc", "cs4"}}, downtime);
            AddConnectivityMetrics(solomonMetrics, metrics.BB4, sensor, solomonDc, {{"network", "bb4"}}, downtime);
            AddConnectivityMetrics(solomonMetrics, metrics.SparseBB, sensor, solomonDc, {{"schedule", "sparse"}, {"network", "bb6"}}, downtime);
            AddConnectivityMetrics(solomonMetrics, metrics.SparseFB, sensor, solomonDc, {{"schedule", "sparse"}, {"network", "fb6"}}, downtime);

            SolomonPusher.RecordMetrics(solomonMetrics);
        }

        TVector<TPacketWindow> PacketWindows;
        TVector<TRttWindow> RttWindows;
        TSolomonPusher& SolomonPusher;
        const TInfraUpdater& InfraUpdater;
        const TInstant GracePeriodEnd;
    };

    TSlaMetricsUpdater::TSlaMetricsUpdater(TSolomonPusher& solomonPusher, const TInfraUpdater& infraUpdater)
        : TSlaMetricsUpdater(solomonPusher, infraUpdater, !!TSettings::Get()->GetNocSlaMetricWindow())
    {
    }

    TSlaMetricsUpdater::TSlaMetricsUpdater(TSolomonPusher& solomonPusher,
                                           const TInfraUpdater& infraUpdater,
                                           bool schedule)
        : TBaseMetricsUpdater(schedule ? YASM_DEFAULT_INTERVAL : TDuration())
        , Impl(MakeHolder<TImpl>(TSettings::Get()->GetNocSlaMetricWindow(), solomonPusher, infraUpdater))
        , SchedulerGuard(schedule ? Schedule() : nullptr)
    {
    }

    TSlaMetricsUpdater::~TSlaMetricsUpdater() = default;

    void TSlaMetricsUpdater::Update() {
        Impl->Update();
    }

    class TCrossDcMetricsUpdater::TImpl: public TNonCopyable {
    public:
        TImpl(TCrossDcCounters& counters,
              const TTopologyStorage& topologyStorage,
              TSolomonPusher& solomonPusher,
              const TInfraUpdater& infraUpdater,
              TDuration window)
            : SolomonPusher(solomonPusher)
            , InfraUpdater(infraUpdater)
            , GracePeriodEnd((2 * window).ToDeadLine())
        {
            auto topology = topologyStorage.GetTopology();

            auto& packetCounterMapList = counters.GetPacketCounterMapList();
            PacketWindowMapList.resize(packetCounterMapList.size());
            for (size_t i = 0; i < packetCounterMapList.size(); ++i) {
                auto& packetCounterMap = packetCounterMapList[i];
                for (const auto& dc : topology->GetDatacenters()) {
                    auto it = packetCounterMap.find(dc->GetReducedId());
                    if (!it.IsEnd()) {
                        PacketWindowMapList[i].emplace(
                            dc->GetName(), TSlidingWindow<TGenericPacketSlaCounters>(window, it->second)
                        );
                    }
                }
            }

            auto& rttCounterMapList = counters.GetRttCounterMapList();
            RttWindowMapList.resize(rttCounterMapList.size());

            for (size_t i = 0; i < rttCounterMapList.size(); ++i) {
                auto& rttCounterMap = rttCounterMapList[i];
                for (const auto& dc : topology->GetDatacenters()) {
                    auto it = rttCounterMap.find(dc->GetReducedId());
                    if (!it.IsEnd()) {
                        RttWindowMapList[i].emplace(
                            dc->GetName(), TSlidingWindow<TGenericRttSlaCounters>(window, it->second)
                        );
                    }
                }
            }
        }

        void Update() {
            if (TInstant::Now() <= GracePeriodEnd) {
                for (auto& packetWindowMap : PacketWindowMapList) {
                    for (auto& item : packetWindowMap) {
                        item.second.Shift();
                    }
                }
                for (auto& rttWindowMap : RttWindowMapList) {
                    for (auto& item : rttWindowMap) {
                        item.second.Shift();
                    }
                }
                return;
            }

            TVector<TSolomonPusher::TMetric> pushMetrics;

            const auto& dcs = TSettings::Get()->GetProbeScheduleDcs();
            for (size_t dcIndex = 0; dcIndex < dcs.size(); ++dcIndex) {
                const auto& sourceDc = dcs[dcIndex];
                bool sourceDowntime = InfraUpdater.IsInfraEventOngoing(sourceDc);
                for (auto& item : PacketWindowMapList[dcIndex]) {
                    auto metrics = item.second.Shift();

                    TString dc = sourceDc + " -> " + item.first;
                    bool downtime = sourceDowntime || InfraUpdater.IsInfraEventOngoing(item.first);

                    AddConnectivityMetrics(pushMetrics, metrics.CS1, "nocsla", dc, {{"tc", "cs1"}}, downtime);
                    AddConnectivityMetrics(pushMetrics, metrics.CS2, "nocsla", dc, {{"tc", "cs2"}}, downtime);
                    AddConnectivityMetrics(pushMetrics, metrics.CS3, "nocsla", dc, {{"tc", "cs3"}}, downtime);
                    AddConnectivityMetrics(pushMetrics, metrics.CS4, "nocsla", dc, {{"tc", "cs4"}}, downtime);
                    AddConnectivityMetrics(pushMetrics, metrics.BB4, "nocsla", dc, {{"network", "bb4"}}, downtime);
                }
                for (auto& item : RttWindowMapList[dcIndex]) {
                    auto metrics = item.second.Shift();

                    TString dc = sourceDc + " -> " + item.first;
                    bool downtime = sourceDowntime || InfraUpdater.IsInfraEventOngoing(item.first);

                    AddRttMetrics(pushMetrics, metrics.CS1, "rtt", dc, {{"tc", "cs1"}}, downtime);
                    AddRttMetrics(pushMetrics, metrics.CS2, "rtt", dc, {{"tc", "cs2"}}, downtime);
                    AddRttMetrics(pushMetrics, metrics.CS3, "rtt", dc, {{"tc", "cs3"}}, downtime);
                    AddRttMetrics(pushMetrics, metrics.CS4, "rtt", dc, {{"tc", "cs4"}}, downtime);
                    AddRttMetrics(pushMetrics, metrics.BB4, "rtt", dc, {{"network", "bb4"}}, downtime);
                }
            }

            SolomonPusher.RecordMetrics(pushMetrics);
        }

    private:
        TVector<THashMap<TString, TSlidingWindow<TGenericPacketSlaCounters>>> PacketWindowMapList;
        TVector<THashMap<TString, TSlidingWindow<TGenericRttSlaCounters>>> RttWindowMapList;
        TSolomonPusher& SolomonPusher;
        const TInfraUpdater& InfraUpdater;
        TInstant GracePeriodEnd;
    };

    TCrossDcMetricsUpdater::TCrossDcMetricsUpdater(TCrossDcCounters& counters,
                                                   const TTopologyStorage& topologyStorage,
                                                   TSolomonPusher& solomonPusher,
                                                   const TInfraUpdater& infraUpdater)
        : TCrossDcMetricsUpdater(counters, topologyStorage, solomonPusher, infraUpdater,
                                 !TSettings::Get()->GetNetmonUrls().empty())
    {
    }

    TCrossDcMetricsUpdater::TCrossDcMetricsUpdater(TCrossDcCounters& counters,
                                                   const TTopologyStorage& topologyStorage,
                                                   TSolomonPusher& solomonPusher,
                                                   const TInfraUpdater& infraUpdater,
                                                   bool schedule)
        : TBaseMetricsUpdater(schedule ? YASM_DEFAULT_INTERVAL : TDuration())
        , Impl(MakeHolder<TImpl>(counters,
                                 topologyStorage,
                                 solomonPusher,
                                 infraUpdater,
                                 TSettings::Get()->GetNocSlaMetricWindow()))
        , SchedulerGuard(schedule ? Schedule() : nullptr)
    {
    }

    TCrossDcMetricsUpdater::~TCrossDcMetricsUpdater() = default;

    void TCrossDcMetricsUpdater::Update() {
        Impl->Update();
    }

    template <class TConfig>
    class TSwitchMetricsUpdater<TConfig>::TImpl {
    public:
        TImpl(TSwitchSlaCounters& perSwitchCounters, const TDuration& window, bool gracePeriod)
            : SwitchCounters(perSwitchCounters)
            , Window(window)
            , AccumulatorMapBox()
            , PastMetricsQueue()
            , GracePeriodEnd(gracePeriod ? (2 * window).ToDeadLine() : TInstant())
            , SwitchCountersSubscription(SwitchCounters.OnMapChanged().Subscribe([this]() {
                UpdateWindowMap();
            }))
        {
            UpdateWindowMap();
        }

        void ShiftWindow() {
            auto accumulatorMap = MakeAtomicShared<TMetricsMap>();

            {
                auto windowMap = WindowMapBox.Own();
                accumulatorMap->reserve(windowMap->size());
                for (auto& item : *windowMap) {
                    (*accumulatorMap)[item.first] = item.second.Shift();
                }
            }

            {
                auto pastMetrics = PastMetricsQueue.Own();
                if (pastMetrics->size() == PastMetricsCount) {
                    pastMetrics->pop();
                }
                pastMetrics->emplace(TInstant::Now(), accumulatorMap);
            }

            AccumulatorMapBox.Swap(accumulatorMap);
        }

        TVector<TPastMetrics> GetPastMetrics(const TInstant& from, const TInstant& to) {
            auto pastMetrics = PastMetricsQueue.Own();
            TVector<TPastMetrics> result;
            for (const auto& item : pastMetrics->Container()) {
                if (item.first > from && item.first <= to &&
                    item.first > GracePeriodEnd)
                {
                    result.emplace_back(item);
                }
            }
            return result;
        }

        TMetricsMapRef GetCurrentMetrics() const {
            return AccumulatorMapBox.Get();
        }


    private:
        using TSlidingWindow = TSlidingWindow<TConfig::template TGenericCounters>;
        using TWindowMap = THashMap<ui64, TSlidingWindow>;

        using TPastMetricsQueue = TQueue<TPastMetrics>;
        using TPastMetricsQueueBox = TPlainLockedBox<TPastMetricsQueue>;

        void UpdateWindowMap() {
            auto counterMap = TConfig::GetCounterMapBox(SwitchCounters);
            auto windowMap = WindowMapBox.Own();

            for (auto& item : *counterMap) {
                typename TWindowMap::insert_ctx ctx = nullptr;
                if (!windowMap->contains(item.first, ctx)) {
                    windowMap->emplace_direct(ctx, item.first,
                                              TSlidingWindow(Window, item.second));
                }
            }
        }

        TSwitchSlaCounters& SwitchCounters;
        TDuration Window;
        TPlainLockedBox<TWindowMap> WindowMapBox;
        TMetricsMapBox AccumulatorMapBox;

        TPastMetricsQueueBox PastMetricsQueue;
        static const size_t PastMetricsCount = 12; // 12 * 5s update interval = 60 seconds

        TInstant GracePeriodEnd;

        TVoidEventHub::TSubscriptionGuard SwitchCountersSubscription;
    };

    template <class TConfig>
    TSwitchMetricsUpdater<TConfig>::TSwitchMetricsUpdater(TSwitchSlaCounters& perSwitchCounters)
        : TSwitchMetricsUpdater(perSwitchCounters,
                                !!TSettings::Get()->GetNocSlaSwitchMapCapacity())
    {
    }

    template <class TConfig>
    TSwitchMetricsUpdater<TConfig>::TSwitchMetricsUpdater(TSwitchSlaCounters& perSwitchCounters,
                                                          bool schedule,
                                                          bool gracePeriod)
        : TBaseMetricsUpdater(schedule ? YASM_DEFAULT_INTERVAL : TDuration())
        , Impl(MakeHolder<TImpl>(perSwitchCounters,
                                 TSettings::Get()->GetNocSlaMetricWindow(),
                                 gracePeriod))
        , SchedulerGuard(schedule ? Schedule() : nullptr)
    {
    }


    template <class TConfig>
    TSwitchMetricsUpdater<TConfig>::~TSwitchMetricsUpdater() {
    }

    template <class TConfig>
    void TSwitchMetricsUpdater<TConfig>::Update() {
        Impl->ShiftWindow();
    }

    template <class TConfig>
    TVector<typename TSwitchMetricsUpdater<TConfig>::TPastMetrics> TSwitchMetricsUpdater<TConfig>::GetPastMetrics(const TInstant& from, const TInstant& to) {
        return Impl->GetPastMetrics(from, to);
    }

    template <class TConfig>
    typename TSwitchMetricsUpdater<TConfig>::TMetricsMapRef TSwitchMetricsUpdater<TConfig>::GetCurrentMetrics() const {
        return Impl->GetCurrentMetrics();
    }

    template class TSwitchMetricsUpdater<TInterSwitchUpdaterConfig>;
    template class TSwitchMetricsUpdater<TInterSwitchRttUpdaterConfig>;
    template class TSwitchMetricsUpdater<TLinkPollerUpdaterConfig>;
}
