#pragma once

#include <infra/netmon/infra.h>
#include <infra/netmon/library/iam.h>
#include <infra/netmon/library/scheduler.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/switch_metrics.h>

#include <library/cpp/monlib/metrics/labels.h>

namespace NNetmon {
    class TSolomonPusher {
    public:
        using TMetric = std::pair<double, NMonitoring::TLabels>;

        static inline TMetric MakeMetric(double value,
                                         const TString& sensor,
                                         const TString& dc,
                                         std::initializer_list<NMonitoring::TLabel> extraLabels) {
            NMonitoring::TLabels labels(extraLabels);
            labels.Add(TStringBuf("sensor"), sensor);
            labels.Add(TStringBuf("dc"), dc);
            return TMetric(value, labels);
        }
        static inline TMetric MakePercentileMetric(double value,
                                                   const TString& sensor,
                                                   const TString& percentile,
                                                   const TString& dc,
                                                   std::initializer_list<NMonitoring::TLabel> extraLabels) {
            NMonitoring::TLabels labels(extraLabels);
            labels.Add(TStringBuf("sensor"), sensor);
            labels.Add(TStringBuf("perc"), percentile);
            labels.Add(TStringBuf("dc"), dc);
            return TMetric(value, labels);
        }

        TSolomonPusher(const TString& pushUrl, const TIamUpdater& iamUpdater);
        ~TSolomonPusher();

        template <class TMetricsContainer>
        void RecordMetrics(const TMetricsContainer& metrics);

    private:
        class TImpl;

        THolder<TImpl> Impl;
        TScheduledTask::TTaskGuard SchedulerGuard;
    };

    // Scheduling the task in TBaseMetricsUpdater constructor leads to a race:
    // child class constructor may not be finished yet by the time Run() is called.
    // So each child class has to declare its own SchedulerGuard, and schedule
    // the task when it's finished initializing.
    class TBaseMetricsUpdater: public TScheduledTask {
    public:
        TBaseMetricsUpdater(TDuration interval)
            : TScheduledTask(interval, true)
        {
        }

        virtual ~TBaseMetricsUpdater() = default;

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                Update();
            });
        }

        virtual void Update() = 0;
    };

    class TCommonMetricsUpdater: public TBaseMetricsUpdater {
    public:
        TCommonMetricsUpdater();

        void Update() override;

    private:
        TScheduledTask::TTaskGuard SchedulerGuard;
    };

    class TSlaMetricsUpdater: public TBaseMetricsUpdater {
    public:
        TSlaMetricsUpdater(TSolomonPusher& solomonPusher, const TInfraUpdater& infraUpdater);
        ~TSlaMetricsUpdater();

        void Update() override;

    private:
        TSlaMetricsUpdater(TSolomonPusher& solomonPusher,
                           const TInfraUpdater& infraUpdater,
                           bool schedule);

        class TImpl;
        THolder<TImpl> Impl;

        TScheduledTask::TTaskGuard SchedulerGuard;
    };

    class TCrossDcMetricsUpdater: public TBaseMetricsUpdater {
    public:
        TCrossDcMetricsUpdater(TCrossDcCounters& counters,
                               const TTopologyStorage& topologyStorage,
                               TSolomonPusher& solomonPusher,
                               const TInfraUpdater& infraUpdater);
        ~TCrossDcMetricsUpdater();

        void Update() override;

    private:
        TCrossDcMetricsUpdater(TCrossDcCounters& counters,
                               const TTopologyStorage& topologyStorage,
                               TSolomonPusher& solomonPusher,
                               const TInfraUpdater& infraUpdater,
                               bool schedule);

        class TImpl;
        THolder<TImpl> Impl;

        TScheduledTask::TTaskGuard SchedulerGuard;
    };

    struct TInterSwitchUpdaterConfig {
        template <class TValue>
        using TGenericCounters = TOrientedPacketSlaCounters<TValue>;

        using TCounterMapBox = TSwitchSlaCounters::TInterSwitchCounterMapBox;

        static inline TCounterMapBox GetCounterMapBox(TSwitchSlaCounters& switchCounters) {
            return switchCounters.GetInterSwitchCounterMapBox();
        }
    };

    struct TInterSwitchRttUpdaterConfig {
        template <class TValue>
        using TGenericCounters = TOrientedRttSlaCounters<TValue>;

        using TCounterMapBox = TSwitchSlaCounters::TInterSwitchRttCounterMapBox;

        static inline TCounterMapBox GetCounterMapBox(TSwitchSlaCounters& switchCounters) {
            return switchCounters.GetInterSwitchRttCounterMapBox();
        }
    };

    struct TLinkPollerUpdaterConfig {
        template <class TValue>
        using TGenericCounters = TGenericPacketSlaCounters<TValue>;

        using TCounterMapBox = TSwitchSlaCounters::TLinkPollerCounterMapBox;

        static inline TCounterMapBox GetCounterMapBox(TSwitchSlaCounters& switchCounters) {
            return switchCounters.GetLinkPollerCounterMapBox();
        }
    };

    template <class TConfig>
    class TSwitchMetricsUpdater: public TBaseMetricsUpdater {
    public:
        using TMetrics = typename TConfig::template TGenericCounters<ui64>;
        using TMetricsMap = THashMap<ui64, TMetrics>;
        using TMetricsMapBox = TAtomicLockedBox<TMetricsMap>;
        using TMetricsMapRef = typename TMetricsMapBox::TConstValueRef;

        using TPastMetrics = std::pair<TInstant, typename TMetricsMapBox::TConstValueRef>;

        explicit TSwitchMetricsUpdater(TSwitchSlaCounters& perSwitchCounters);
        TSwitchMetricsUpdater(TSwitchSlaCounters& perSwitchCounters,
                              bool schedule,
                              bool gracePeriod = true);
        ~TSwitchMetricsUpdater();

        void Update() override;
        // Returns metrics in the interval (from, to]
        TVector<TPastMetrics> GetPastMetrics(const TInstant& from, const TInstant& to);
        TMetricsMapRef GetCurrentMetrics() const;

    private:
        class TImpl;
        THolder<TImpl> Impl;

        TScheduledTask::TTaskGuard SchedulerGuard;
    };

    using TInterSwitchMetricsUpdater = TSwitchMetricsUpdater<TInterSwitchUpdaterConfig>;
    using TInterSwitchRttMetricsUpdater = TSwitchMetricsUpdater<TInterSwitchRttUpdaterConfig>;
    using TLinkPollerMetricsUpdater = TSwitchMetricsUpdater<TLinkPollerUpdaterConfig>;
}
