#include <infra/netmon/monitors.h>
#include <infra/netmon/schema_maintainer.h>
#include <infra/netmon/settings.h>

#include <infra/netmon/library/clickhouse/client.h>
#include <infra/netmon/library/clickhouse/translate.h>
#include <infra/netmon/library/settings.h>

#include <util/string/subst.h>

namespace NNetmon {
    namespace {
        const char* QUERY_TEMPLATE = R"(
SELECT
    database,
    table
FROM system.replicas
WHERE match(table, '^{tablePrefix}') AND (
       is_readonly
    OR is_session_expired
    OR future_parts > 20
    OR parts_to_check > 10
    OR queue_size > 20
    OR inserts_in_queue > 10
    OR log_max_index - log_pointer > 10
)
)";
    }

    class TClickhouseMonitor::TImpl : public TScheduledTask {
    public:
        TImpl(bool schedule)
            : TScheduledTask(TSettings::Get()->GetMonitorInterval())
            , Ready(!schedule)
        {
        }

        inline TThreadPool::TFuture Run() override {
            TString query(QUERY_TEMPLATE);
            SubstGlobal(query, "{tablePrefix}", PROBES_TABLE_PREFIX);
            TClickhouseClient::TQueryOptions options(query);
            return TClickhouseClient::Get()->Select(options, [this] (const NClickHouse::TBlock& block) {
                const auto rows(TranslateClickhouseBlock<std::tuple<TString, TString>>(block));
                AtomicSet(Ready, rows.empty());
            }).Apply([this](const TClickhouseClient::TFuture& future) {
                try {
                    future.GetValue(TDuration::Max());
                } catch(...) {
                    AtomicSet(Ready, false);
                    throw;
                }
            });
        }

        inline bool IsReady() const {
            return AtomicGet(Ready);
        }

    private:
        TAtomic Ready;
    };

    TClickhouseMonitor::TClickhouseMonitor()
        : TClickhouseMonitor(!TLibrarySettings::Get()->GetClickHouseShards().empty())
    {
    }

    TClickhouseMonitor::TClickhouseMonitor(bool schedule)
        : Impl(MakeHolder<TImpl>(schedule))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TClickhouseMonitor::~TClickhouseMonitor() {
    }

    bool TClickhouseMonitor::IsReady() const {
        return Impl->IsReady();
    }
}
