#include <infra/netmon/network_availability.h>
#include <infra/netmon/state_index.h>

namespace NNetmon {
    void TNetworkAvailability::AddState(const TSwitchPairState& state, const TTopologySelector& selector) {
        if (state.GetSource().GetDatacenter() != state.GetTarget().GetDatacenter()) {
            return;
        }

        auto weight(state.GetWeight(selector));

        // source is alive if it can send probes
        auto sourceIt = SwitchMap.find(state.GetSourceRef());
        if (sourceIt.IsEnd()) {
            sourceIt = SwitchMap.emplace(state.GetSourceRef(), TAvailabilityState()).first;
        }
        sourceIt->second.Alive.Append(state.IsSourceAlive(), weight);

        // target is alive if it's reachable in probes
        auto targetIt = SwitchMap.find(state.GetTargetRef());
        if (targetIt.IsEnd()) {
            targetIt = SwitchMap.emplace(state.GetTargetRef(), TAvailabilityState()).first;
        }
        targetIt->second.Alive.Append(state.IsTargetAlive(), weight);

        targetIt->second.Connectivity.Merge(state.GetConnectivityHistogram(), weight);
    }

    void TNetworkAvailability::MergeFrom(const TNetworkAvailability& other) {
        for (const auto& pair : other.SwitchMap) {
            auto it = SwitchMap.find(pair.first);
            if (it.IsEnd()) {
                SwitchMap.emplace(pair.first, pair.second);
            } else {
                it->second.Merge(pair.second);
            }
        }
    }

    void TNetworkAvailability::FillMap(TSwitchMap& switchMap) const {
        for (const auto& pair : SwitchMap) {
            switchMap[pair.first] = pair.second;
        }
    }

    void TNetworkAvailability::FillMap(TLineMap& queueMap) const {
        for (const auto& pair : SwitchMap) {
            queueMap[pair.first.GetLine()].Merge(pair.second);
        }
    }

    void TNetworkAvailability::FillMap(TDatacenterMap& datacenterMap) const {
        for (const auto& pair : SwitchMap) {
            datacenterMap[pair.first.GetDatacenter()].Merge(pair.second);
        }
    }

    void TNetworkAvailability::FromProto(
            const TTopologyStorage& topologyStorage,
            const TNetworkAvailability::TFlatState& states) {
        for (const auto& state : states) {
            auto switch_(topologyStorage.FindSwitch(*state->Switch()));
            if (switch_) {
                SwitchMap[switch_].FromProto(*state);
            }
        }
    }

    flatbuffers::Offset<TNetworkAvailability::TFlatState> TNetworkAvailability::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {

        std::vector<flatbuffers::Offset<NAggregation::TSwitchAvailability>> states;
        states.reserve(SwitchMap.size());
        for (const auto& pair : SwitchMap) {
            auto alive(pair.second.Alive.ToProto());
            states.emplace_back(NAggregation::CreateTSwitchAvailability(
                builder,
                &pair.first->ToProto(),
                &alive,
                pair.second.Connectivity.ToProto(builder)
            ));
        }
        return builder.CreateVector(states);
    }
}
