#include <infra/netmon/probe.h>

#include <util/generic/algorithm.h>

namespace NNetmon {
    TAtomic MemPoolStatsProbeCount;

    void TProbe::Out(IOutputStream& stream) const {
        stream << "TProbe(source=" << SourceIface->GetName() << ", target=" << TargetIface->GetName()
               << ", score=" << Score << ", rtt=" << RoundTripTime << ")";
    }

    bool TProbe::HasExpressionId(const TTopologySelector& selector, TExpressionId expressionId) const {
        const auto* sourceExpressions(selector.FindSourceExpressions(SourceIface));
        const auto* targetExpressions(selector.FindTargetExpressions(TargetIface));
        if (!sourceExpressions || !targetExpressions) {
            return false;
        }
        return (
            BinarySearch(sourceExpressions->begin(), sourceExpressions->end(), expressionId)
            && BinarySearch(targetExpressions->begin(), targetExpressions->end(), expressionId));
    }

    void TProbe::IntersectExpressionIds(const TTopologySelector& selector, TExpressionIdList& expressionIds) const {
        const auto* sourceExpressions(selector.FindSourceExpressions(SourceIface));
        const auto* targetExpressions(selector.FindTargetExpressions(TargetIface));
        if (!sourceExpressions || !targetExpressions) {
            return;
        }
        expressionIds.reserve(Max(sourceExpressions->size(), targetExpressions->size()));
        SetIntersection(sourceExpressions->begin(), sourceExpressions->end(),
                        targetExpressions->begin(), targetExpressions->end(),
                        std::back_inserter(expressionIds));
    }

    flatbuffers::Offset<NProbe::TProbe> TProbe::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        return NProbe::CreateTProbe(
            builder,
            &SourceIface->ToProto(),
            &TargetIface->ToProto(),
            &SourceAddress.ToProto(),
            &TargetAddress.ToProto(),
            SourcePort,
            TargetPort,
            Generated.Seconds(),
            Score,
            RoundTripTime
        );
    }

    void TProbeSectionKey::Out(IOutputStream& stream) const {
        stream << "TProbeSectionKey(network=" << GetNetwork()
               << ", protocol=" << GetProtocol()
               << ")";
    }

    void TProbeSliceKey::Out(IOutputStream& stream) const {
        stream << "TProbeSliceKey(shard=" << GetShardIndex()
               << ", network=" << GetNetwork()
               << ", protocol=" << GetProtocol()
               << ")";
    }

    TProbeAggregatorKey::TProbeAggregatorKey(TExpressionId expressionId, ENetworkType networkType, EProtocolType protocolType)
        : tuple{
            expressionId,
            networkType,
            protocolType
        }
        , Flat(
            GetExpressionId(),
            static_cast<NCommon::ENetwork>(GetNetwork()),
            static_cast<NCommon::EProtocol>(GetProtocol())
        )
    {
    }

    TProbeAggregatorKey::TProbeAggregatorKey(const NCommon::TAggregatorKey& key)
        : TProbeAggregatorKey(
            key.ExpressionId(),
            static_cast<ENetworkType>(key.Network()),
            static_cast<EProtocolType>(key.Protocol())
        )
    {
    }

    const NCommon::TAggregatorKey& TProbeAggregatorKey::ToProto() const {
        return Flat;
    }

    void TProbeAggregatorKey::Out(IOutputStream& stream) const {
        stream << "TProbeAggregatorKey(expressionId=" << GetExpressionId()
               << ", network=" << GetNetwork()
               << ", protocol=" << GetProtocol()
               << ")";
    }
}

template <>
void Out<NNetmon::TProbe>(IOutputStream& stream,
                          TTypeTraits<NNetmon::TProbe>::TFuncParam probe) {
    probe.Out(stream);
}

template <>
void Out<NNetmon::TProbeSectionKey>(IOutputStream& stream,
                                    TTypeTraits<NNetmon::TProbeSectionKey>::TFuncParam key) {
    key.Out(stream);
}

template <>
void Out<NNetmon::TProbeSliceKey>(IOutputStream& stream,
                                  TTypeTraits<NNetmon::TProbeSliceKey>::TFuncParam key) {
    key.Out(stream);
}

template <>
void Out<NNetmon::TProbeAggregatorKey>(IOutputStream& stream,
                                       TTypeTraits<NNetmon::TProbeAggregatorKey>::TFuncParam key) {
    key.Out(stream);
}
