#include <infra/netmon/probe_aggregator.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    namespace {
        template <class T, class TProcessor>
        class TTaskRunner : public TScheduledTask {
        public:
            TTaskRunner(const TDuration& interval, TProbeAggregator& aggregator)
                : TScheduledTask(interval)
                , Aggregator(aggregator)
            {
            }

            TThreadPool::TFuture Run() override {
                return TThreadPool::Get()->Add([this]() {
                    TSimpleTimer timer;
                    Processor(Aggregator);
                    DEBUG_LOG << Aggregator << " processing with " << TypeName<TProcessor>()
                              << " took " << timer.Get() << Endl;
                });
            }

        private:
            TProbeAggregator& Aggregator;
            TProcessor Processor;
        };

        class TSwitchTask: public TTaskRunner<TSwitchTask, TProbeAggregator::TSwitchProcessor> {
            using TTaskRunner::TTaskRunner;
        };

        class TLineTask: public TTaskRunner<TLineTask, TProbeAggregator::TLineProcessor> {
            using TTaskRunner::TTaskRunner;
        };

        class TDatacenterTask: public TTaskRunner<TDatacenterTask, TProbeAggregator::TDatacenterProcessor> {
            using TTaskRunner::TTaskRunner;
        };
    }

    TProbeAggregator::TProbeAggregator(const TProbeAggregatorKey& key,
                                       const TSliceCollector& sliceCollector)
        : Key{key}
        , SliceCollector(sliceCollector)
        , SwitchTask(MakeHolder<TSwitchTask>(TSettings::Get()->GetSwitchAggregationInterval(), *this))
        , QueueTask(MakeHolder<TLineTask>(TSettings::Get()->GetLineAggregationInterval(), *this))
        , DatacenterTask(MakeHolder<TDatacenterTask>(TSettings::Get()->GetDcAggregationInterval(), *this))
        , SwitchTaskGuard(SwitchTask->Schedule())
        , QueueTaskGuard(QueueTask->Schedule())
        , DatacenterTaskGuard(DatacenterTask->Schedule())
    {
    }

    void TProbeAggregator::TSwitchProcessor::operator()(TProbeAggregator& aggregator) {
        const auto index(aggregator.GetSwitchIndex());
        if (index) {
            TUnistat::Instance().PushSignalUnsafe(
                ENetmonSignals::StateSwitchAge,
                (TInstant::Now() - index->GetGenerated()).MilliSeconds()
            );
        }
    }

    void TProbeAggregator::TLineProcessor::operator()(TProbeAggregator& aggregator) {
        const auto index(aggregator.GetLineIndex());
        if (index) {
            TUnistat::Instance().PushSignalUnsafe(
                ENetmonSignals::StateQueueAge,
                (TInstant::Now() - index->GetGenerated()).MilliSeconds()
            );
        }
    }

    void TProbeAggregator::TDatacenterProcessor::operator()(TProbeAggregator& aggregator) {
        const auto index(aggregator.GetDatacenterIndex());
        if (index) {
            TUnistat::Instance().PushSignalUnsafe(
                ENetmonSignals::StateDcAge,
                (TInstant::Now() - index->GetGenerated()).MilliSeconds()
            );
        }
    }

    void TProbeAggregator::Out(IOutputStream& stream) const {
        stream << "TProbeAggregator(key=" << GetKey() << ")";
    }
}

template <>
void Out<NNetmon::TProbeAggregator>(IOutputStream& stream,
                                    TTypeTraits<NNetmon::TProbeAggregator>::TFuncParam group) {
    group.Out(stream);
}
