#pragma once

#include <infra/netmon/slice_collector.h>
#include <infra/netmon/library/scheduler.h>

#include <library/cpp/containers/intrusive_rb_tree/rb_tree.h>

namespace NNetmon {
    class TProbeProcessor;

    namespace {
        struct TProbeAggregatorCompare {
            template <class T>
            static inline bool Compare(const T& lhs, const T& rhs) {
                return lhs < rhs;
            }

            template <class T>
            static inline bool Compare(const typename T::TKey& lhs, const T& rhs) {
                return lhs < rhs.GetKey();
            }

            template <class T>
            static inline bool Compare(const T& lhs, const typename T::TKey& rhs) {
                return lhs.GetKey() < rhs;
            }
        };
    }

    class TProbeAggregator: public TRbTreeItem<TProbeAggregator, TProbeAggregatorCompare>,
                            public TAtomicRefCount<TProbeAggregator>,
                            public TNonCopyable {
    public:
        using TRef = TIntrusivePtr<TProbeAggregator>;
        using TConstRef = TIntrusiveConstPtr<TProbeAggregator>;
        using TRefVector = TVector<TProbeAggregator::TConstRef>;

        using TKey = TProbeAggregatorKey;
        using TTree = TRbTree<TProbeAggregator, TProbeAggregatorCompare>;

        struct TSwitchProcessor {
            void operator()(TProbeAggregator& aggregator);
        };

        struct TLineProcessor {
            void operator()(TProbeAggregator& aggregator);
        };

        struct TDatacenterProcessor {
            void operator()(TProbeAggregator& aggregator);
        };

        TProbeAggregator(const TProbeAggregatorKey& key,
                         const TSliceCollector& sliceCollector);

        const TProbeAggregatorKey& GetKey() const {
            return Key;
        }

        TSwitchPairIndex::TRef GetSwitchIndex() const {
            return SliceCollector.GetSwitchIndex(Key);
        }
        TLinePairIndex::TRef GetLineIndex() const {
            return SliceCollector.GetLineIndex(Key);
        }
        TDatacenterPairIndex::TRef GetDatacenterIndex() const {
            return SliceCollector.GetDatacenterIndex(Key);
        }

        void Out(IOutputStream& stream) const;

        inline bool operator<(const TProbeAggregator& rhs) const {
            return Key < rhs.Key;
        }

    private:
        const TProbeAggregatorKey Key;
        const TSliceCollector& SliceCollector;

        THolder<TScheduledTask> SwitchTask;
        THolder<TScheduledTask> QueueTask;
        THolder<TScheduledTask> DatacenterTask;

        TScheduledTask::TTaskGuard SwitchTaskGuard;
        TScheduledTask::TTaskGuard QueueTaskGuard;
        TScheduledTask::TTaskGuard DatacenterTaskGuard;
    };
}
