#include <infra/netmon/probe_aggregator_maintainer.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    class TProbeAggregatorMaintainer::TImpl : public TScheduledTask {
    public:
        TImpl(const TSliceCollector& sliceCollector)
            : TScheduledTask(TSettings::Get()->GetDcAggregationInterval())
            , SliceCollector(sliceCollector)
        {
        }

        ~TImpl() {
            auto tree(AggregatorTree.Own());
            while (!tree->Empty()) {
                TProbeAggregator::TRef aggregator(&(*tree->Begin()));
                aggregator->UnRef();
            }
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                TProbeAggregatorKeySet keys;
                SliceCollector.FillKeys(keys);

                auto tree(AggregatorTree.Own());

                // remove those aggregators that aren't exists
                for (auto it(tree->Begin()); it != tree->End();) {
                    if (keys.contains(it->GetKey())) {
                        keys.erase(it->GetKey());
                        ++it;
                    } else {
                        TProbeAggregator::TRef tag(&(*it));
                        ++it;
                        tag->UnLink();
                        tag->UnRef();
                        // deletion may wait for futures
                        auto unguard = Unguard(tree.GetGuard());
                        tag.Drop();
                    }
                }

                // and create all that left
                for (const auto& key : keys) {
                    TProbeAggregator::TRef aggregator(MakeIntrusive<TProbeAggregator>(key, SliceCollector));
                    aggregator->Ref();
                    tree->Insert(aggregator.Get());
                }
            }, false);
        }

        TProbeAggregator::TRef GetAggregator(const TProbeAggregatorKey &key) const {
            auto tree(AggregatorTree.Own());
            auto* foundAggregator = tree->Find(key);
            if (foundAggregator == nullptr) {
                ythrow yexception() << "aggregator not found";
            } else {
                return &(*foundAggregator);
            }
        }

        TProbeAggregator::TRefVector GetAggregators() const {
            auto tree(AggregatorTree.Own());
            TProbeAggregator::TRefVector aggregators;
            for (auto it(tree->Begin()); it != tree->End(); ++it) {
                aggregators.emplace_back(&(*it));
            }
            return aggregators;
        }

        TProbeAggregator::TRefVector FindAggregators(TExpressionId expressionId) const {
            TProbeAggregator::TRefVector aggregators;
            auto tree(AggregatorTree.Own());
            const TProbeAggregatorKey key{expressionId, ENetworkType::NIL_NETWORK, EProtocolType::NIL_PROTOCOL};
            for (TProbeAggregator::TTree::TIterator it(tree->LowerBound(key)); it != tree->End(); ++it) {
                if (it->GetKey().GetExpressionId() == expressionId) {
                    aggregators.emplace_back(&(*it));
                } else {
                    break;
                }
            }
            return aggregators;
        }

        bool ExpressionExists(TExpressionId expressionId) const {
            auto tree(AggregatorTree.Own());
            const TProbeAggregatorKey key{expressionId, ENetworkType::NIL_NETWORK, EProtocolType::NIL_PROTOCOL};
            for (TProbeAggregator::TTree::TConstIterator it(tree->LowerBound(key)); it != tree->End(); ++it) {
                if (it->GetKey().GetExpressionId() == expressionId) {
                    return true;
                } else {
                    return false;
                }
            }
            return false;
        }

    private:
        const TSliceCollector& SliceCollector;

        TPlainLockedBox<TProbeAggregator::TTree> AggregatorTree;
    };

    TProbeAggregatorMaintainer::TProbeAggregatorMaintainer(const TSliceCollector& sliceCollector)
        : Impl(MakeHolder<TImpl>(sliceCollector))
        , SchedulerGuard(Impl->Schedule())
    {
    }

    TProbeAggregatorMaintainer::~TProbeAggregatorMaintainer() {
    }

    TProbeAggregator::TRef TProbeAggregatorMaintainer::GetAggregator(const TProbeAggregatorKey &key) const {
        return Impl->GetAggregator(key);
    }

    TProbeAggregator::TRefVector TProbeAggregatorMaintainer::GetAggregators() const {
        return Impl->GetAggregators();
    }

    TProbeAggregator::TRefVector TProbeAggregatorMaintainer::FindAggregators(TExpressionId expressionId) const {
        return Impl->FindAggregators(expressionId);
    }

    bool TProbeAggregatorMaintainer::ExpressionExists(TExpressionId expressionId) const {
        return Impl->ExpressionExists(expressionId);
    }
}
