#include <infra/netmon/probe_dumper.h>
#include <infra/netmon/schema_maintainer.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/library/api_client_helpers.h>
#include <infra/netmon/library/settings.h>

#include <infra/netmon/library/clickhouse/client.h>

#include <library/cpp/clickhouse/client/columns/date.h>
#include <library/cpp/clickhouse/client/columns/numeric.h>
#include <library/cpp/clickhouse/client/columns/string.h>

#include <util/string/builder.h>
#include <util/generic/xrange.h>

namespace NNetmon {
    TAtomic MemPoolStatsBatchCount;

    namespace {
        const TDuration QUERY_TIMEOUT = TDuration::Seconds(10);

        inline TString GetCurrentProbeTable(const TInstant& now) {
            struct tm timestamp;
            now.LocalTime(&timestamp);
            return TStringBuilder() << PROBES_TABLE_PREFIX << Strftime(PROBES_TABLE_FORMAT, &timestamp);
        }

    } // end namespace

    flatbuffers::Offset<NProbe::TProbe> TDumperProbe::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        return NProbe::CreateTProbe(
            builder,
            &SourceIface->ToProto(),
            &TargetIface->ToProto(),
            &SourceAddress.ToProto(),
            &TargetAddress.ToProto(),
            SourcePort,
            TargetPort,
            Generated.Seconds(),
            Score,
            RoundTripTime,
            static_cast<NCommon::ENetwork>(Network),
            static_cast<NCommon::EProtocol>(Protocol)
        );
    }

    TThreadPool::TFuture TClickhouseBatch::Dispatch() {
        ResetTimer();

        const auto& records(GetRecords());

        TVector<ui64> sourceFqdns(records.Size());
        TVector<ui64> sourceSwitchs(records.Size());
        TVector<ui64> sourceQueues(records.Size());
        TVector<ui64> sourceDcs(records.Size());
        TVector<TString> sourceAddresses(records.Size());
        TVector<ui16> sourcePorts(records.Size());

        TVector<ui64> targetFqdns(records.Size());
        TVector<ui64> targetSwitchs(records.Size());
        TVector<ui64> targetQueues(records.Size());
        TVector<ui64> targetDcs(records.Size());
        TVector<TString> targetAddresses(records.Size());
        TVector<ui16> targetPorts(records.Size());

        TVector<ui8> networks(records.Size());
        TVector<ui8> protocols(records.Size());
        TVector<double> scores(records.Size());
        TVector<double> rtts(records.Size());

        auto generated(NClickHouse::TColumnDateTime::Create());
        auto inserted(NClickHouse::TColumnDateTime::Create());

        TInstant now(TInstant::Now());
        for (auto index : xrange(records.Size())) {
            const auto& probe(records[index]->GetObject());
            const auto& sourceIface = probe.GetSourceIface();
            const auto& targetIface = probe.GetTargetIface();

            sourceFqdns[index] = sourceIface->GetReducedId();
            sourceSwitchs[index] = sourceIface->GetSwitch().GetReducedId();
            sourceQueues[index] = sourceIface->GetLine().GetReducedId();
            sourceDcs[index] = sourceIface->GetDatacenter().GetReducedId();
            sourceAddresses[index] = probe.GetSourceAddress().ToBytes();
            sourcePorts[index] = probe.GetSourcePort();

            targetFqdns[index] = targetIface->GetReducedId();
            targetSwitchs[index] = targetIface->GetSwitch().GetReducedId();
            targetQueues[index] = targetIface->GetLine().GetReducedId();
            targetDcs[index] = targetIface->GetDatacenter().GetReducedId();
            targetAddresses[index] = probe.GetTargetAddress().ToBytes();
            targetPorts[index] = probe.GetTargetPort();

            networks[index] = ui8(probe.GetNetwork());
            protocols[index] = ui8(probe.GetProtocol());
            scores[index] = probe.GetScore();
            rtts[index] = probe.GetRoundTripTime() > 0.0 ? probe.GetRoundTripTime() : -1.0;

            generated->Append(probe.GetGenerated());
            inserted->Append(now);
        }

        NClickHouse::TBlock block;

        block.AppendColumn("SourceFqdn", NClickHouse::TColumnUInt64::Create(sourceFqdns));
        block.AppendColumn("SourceSwitch", NClickHouse::TColumnUInt64::Create(sourceSwitchs));
        block.AppendColumn("SourceQueue", NClickHouse::TColumnUInt64::Create(sourceQueues));
        block.AppendColumn("SourceDc", NClickHouse::TColumnUInt64::Create(sourceDcs));
        block.AppendColumn("SourceAddress", NClickHouse::TColumnString::Create(sourceAddresses));
        block.AppendColumn("SourcePort", NClickHouse::TColumnUInt16::Create(sourcePorts));

        block.AppendColumn("TargetFqdn", NClickHouse::TColumnUInt64::Create(targetFqdns));
        block.AppendColumn("TargetSwitch", NClickHouse::TColumnUInt64::Create(targetSwitchs));
        block.AppendColumn("TargetQueue", NClickHouse::TColumnUInt64::Create(targetQueues));
        block.AppendColumn("TargetDc", NClickHouse::TColumnUInt64::Create(targetDcs));
        block.AppendColumn("TargetAddress", NClickHouse::TColumnString::Create(targetAddresses));
        block.AppendColumn("TargetPort", NClickHouse::TColumnUInt16::Create(targetPorts));

        block.AppendColumn("Network", NClickHouse::TColumnUInt8::Create(networks));
        block.AppendColumn("Protocol", NClickHouse::TColumnUInt8::Create(protocols));
        block.AppendColumn("Score", NClickHouse::TColumnFloat64::Create(scores));
        block.AppendColumn("Rtt", NClickHouse::TColumnFloat64::Create(rtts));

        block.AppendColumn("Generated", generated);
        block.AppendColumn("Inserted", inserted);

        return TClickhouseClient::Get()->Insert(
            TClickhouseClient::TQueryOptions(GetCurrentProbeTable(now))
                .SetTimeout(QUERY_TIMEOUT)
                .SetShardIndex(GetShardIndex()),
            std::move(block)
        );
    }

    TThreadPool::TFuture TSlicerBatch::Dispatch() {
        ResetTimer();

        flatbuffers::FlatBufferBuilder builder;
        std::vector<flatbuffers::Offset<NProbe::TProbe>> probes;
        for (const auto& record : GetRecords()) {
            probes.push_back(record->GetObject().ToProto(builder));
        }

        const auto& shard = TSettings::Get()->GetSlicerShards()[GetShardIndex()];
        builder.Finish(NApi::CreateTAppendProbesRequest(
            builder,
            builder.CreateVector(probes)
        ));

        return MakeRequest(shard, "/slicer/v1/append_probes", builder).Apply(
            [] (const TNehRequester::TFuture& future_) {
                future_.GetValue();
            }
        );
    }

    std::size_t TClickhouseDumper::GetShardCount() {
        return TLibrarySettings::Get()->GetClickHouseShards().size();
    }

    std::size_t TClickhouseDumper::GetBatchSize() {
        return 1000000;
    }

    TClickhouseDumper::TBatch::TRef TClickhouseDumper::CreateBatch(std::size_t shardIndex) {
        return MakeHolder<TBatch>(shardIndex, GetBatchSize());
    }

    std::size_t TSlicerDumper::GetShardCount() {
        return TSettings::Get()->GetSlicerShards().size();
    }

    std::size_t TSlicerDumper::GetBatchSize() {
        return 100000;
    }

    TSlicerDumper::TBatch::TRef TSlicerDumper::CreateBatch(std::size_t shardIndex) {
        return MakeHolder<TBatch>(shardIndex, GetBatchSize());
    }
}
