#pragma once

#include <infra/netmon/library/record_dumper.h>
#include <infra/netmon/topology/topology_storage.h>
#include <infra/netmon/topology/types.h>

#include <infra/netmon/idl/probe.fbs.h>
#include <infra/netmon/idl/api.fbs.h>

namespace NNetmon {
    class TDumperProbe {
    public:
        explicit TDumperProbe(
                const TTopology::THostInterfaceRef& sourceIface,
                const TTopology::THostInterfaceRef& targetIface,
                const TIpAddress& sourceAddress,
                const TIpAddress& targetAddress,
                ui16 sourcePort,
                ui16 targetPort,
                ENetworkType network,
                EProtocolType protocol,
                double score,
                double roundTripTime,
                ui64 generated)
            : SourceIface(sourceIface)
            , TargetIface(targetIface)
            , SourceAddress(sourceAddress)
            , TargetAddress(targetAddress)
            , SourcePort(sourcePort)
            , TargetPort(targetPort)
            , Network(network)
            , Protocol(protocol)
            , Score(score)
            , RoundTripTime(roundTripTime)
            , Generated(TInstant::Seconds(generated))
        {
        }

        inline const TTopology::THostInterfaceRef& GetSourceIface() const {
            return SourceIface;
        }
        inline const TTopology::THostInterfaceRef& GetTargetIface() const {
            return TargetIface;
        }
        inline const TIpAddress& GetSourceAddress() const {
            return SourceAddress;
        }
        inline const TIpAddress& GetTargetAddress() const {
            return TargetAddress;
        }
        inline ui16 GetSourcePort() const {
            return SourcePort;
        }
        inline ui16 GetTargetPort() const {
            return TargetPort;
        }
        inline ENetworkType GetNetwork() const {
            return Network;
        }
        inline EProtocolType GetProtocol() const {
            return Protocol;
        }
        inline double GetScore() const {
            return Score;
        }
        inline double GetRoundTripTime() const {
            return RoundTripTime;
        }
        inline const TInstant& GetGenerated() const {
            return Generated;
        }

        flatbuffers::Offset<NProbe::TProbe> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        const TTopology::THostInterfaceRef SourceIface;
        const TTopology::THostInterfaceRef TargetIface;

        const TIpAddress SourceAddress;
        const TIpAddress TargetAddress;

        const ui16 SourcePort;
        const ui16 TargetPort;

        const ENetworkType Network;
        const EProtocolType Protocol;

        const double Score;
        const double RoundTripTime;
        const TInstant Generated;

        TInstant Deadline;
    };

    class TDumperProbeBatch: public TBatch<TDumperProbe> {
    public:
        using TBatch<TDumperProbe>::TBatch;

        using TRecordQueue = TRecord::TRecordQueue;
        using TRecordVector = TRecord::TRefVector;
    };

    class TClickhouseBatch: public TDumperProbeBatch {
    public:
        using TDumperProbeBatch::TDumperProbeBatch;

        TThreadPool::TFuture Dispatch() override;
    };

    class TSlicerBatch: public TDumperProbeBatch {
    public:
        using TDumperProbeBatch::TDumperProbeBatch;

        TThreadPool::TFuture Dispatch() override;
    };

    class TClickhouseDumper: public TRecordDumper<TClickhouseBatch> {
    public:
        using TRecordDumper<TClickhouseBatch>::TRecordDumper;

        std::size_t GetShardCount() override;
        std::size_t GetBatchSize() override;
        TBatch::TRef CreateBatch(std::size_t shardIndex) override;
    };

    class TSlicerDumper: public TRecordDumper<TSlicerBatch> {
    public:
        using TRecordDumper<TSlicerBatch>::TRecordDumper;

        std::size_t GetShardCount() override;
        std::size_t GetBatchSize() override;
        TBatch::TRef CreateBatch(std::size_t shardIndex) override;
    };
}

template <>
class THash<NNetmon::TDumperProbe> {
public:
    inline size_t operator()(const NNetmon::TDumperProbe& probe) const {
        return probe.GetTargetIface()->GetSwitch().GetHash();
    }
};
