#pragma once

#include <infra/netmon/library/boxes.h>
#include <infra/netmon/probe_slice.h>

namespace NNetmon {
    class TProbeGatherer : public TAtomicRefCount<TProbeGatherer> {
    public:
        using TRef = TIntrusivePtr<TProbeGatherer>;

        struct TQuery {
            TMaybe<TSwitchPairKey> SwitchKey;
            TMaybe<TLinePairKey> QueueKey;
            TMaybe<TDatacenterPairKey> DatacenterKey;
        };

        template <typename... Args>
        static inline TRef Make(Args&&... args) {
            return new TProbeGatherer(std::forward<Args>(args)...);
        }

        template <class TAccessor>
        NThreading::TFuture<TProbe::TRefVector> FindProbes(const TQuery& query) const {
            TVector<TThreadPool::TFuture> futures;
            for (const auto& storage : ProbeStorages) {
                futures.emplace_back(TThreadPool::Get()->Add([this, storage, query]() {
                    const auto part(FindAndFilterProbes<TAccessor>(*storage, query));
                    auto result(ResultBox.Own());
                    result->insert(result->end(), part.begin(), part.end());
                }));
            }

            TRef self(const_cast<TProbeGatherer*>(this));
            return NThreading::WaitExceptionOrAll(futures).Apply([self](const TThreadPool::TFuture& future) {
                future.GetValue();
                return *self->ResultBox.Own();
            });
        }

    private:
        struct TFilter {
            inline bool operator()(const TProbe* probe) const noexcept {
                for (const auto& filter : Filters) {
                    if (!filter(probe)) {
                        return false;
                    }
                }
                return true;
            }

            std::vector<std::function<bool(const TProbe*)>> Filters;
        };

        TProbeGatherer(const TProbeSlice::TRefVector& slices, TExpressionId expressionId, const TTopologyStorage& topologyStorage)
            : ExpressionId(expressionId)
            , TopologyStorage(topologyStorage)
        {
            ProbeStorages.reserve(slices.size());
            for (const auto* slice : slices) {
                ProbeStorages.push_back(slice->GetProbeStorage());
            }
        }

        template <class TAccessor>
        TProbe::TRefVector FindAndFilterProbes(TProbeStorage& storage, const TQuery& query) const {
            TFilter filter;

            auto selector(TopologyStorage.GetTopologySelector());
            filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                return probe->HasExpressionId(*selector, ExpressionId);
            });

            if (query.SwitchKey && query.SwitchKey->IsValid()) {
                return storage.FindProbes<TAccessor>(*query.SwitchKey, filter);

            } else if (query.QueueKey && query.QueueKey->IsValid()) {
                if (query.SwitchKey && query.SwitchKey->IsPartiallyValid()) {
                    if (query.SwitchKey->GetSource()) {
                        filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                            return *query.SwitchKey->GetSource() == probe->GetSourceIface()->GetSwitch();
                        });
                    } else {
                        filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                            return *query.SwitchKey->GetTarget() == probe->GetTargetIface()->GetSwitch();
                        });
                    }
                }
                return storage.FindProbes<TAccessor>(*query.QueueKey, filter);

            } else if (query.DatacenterKey && query.DatacenterKey->IsValid()) {
                if (query.QueueKey && query.QueueKey->IsPartiallyValid()) {
                    if (query.QueueKey->GetSource()) {
                        filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                            return *query.QueueKey->GetSource() == probe->GetSourceIface()->GetLine();
                        });
                    } else {
                        filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                            return *query.QueueKey->GetTarget() == probe->GetTargetIface()->GetLine();
                        });
                    }
                }
                return storage.FindProbes<TAccessor>(*query.DatacenterKey, filter);

            } else {
                if (query.DatacenterKey && query.DatacenterKey->IsPartiallyValid()) {
                    if (query.DatacenterKey->GetSource()) {
                        filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                            return *query.DatacenterKey->GetSource() == probe->GetSourceIface()->GetDatacenter();
                        });
                    } else {
                        filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                            return *query.DatacenterKey->GetTarget() == probe->GetTargetIface()->GetDatacenter();
                        });
                    }

                    /* Require partial dc key intentionally here */
                    if (query.QueueKey && query.QueueKey->IsPartiallyValid()) {
                        if (query.QueueKey->GetSource()) {
                            filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                                return *query.QueueKey->GetSource() == probe->GetSourceIface()->GetLine();
                            });
                        } else {
                            filter.Filters.emplace_back([&](const TProbe* probe) noexcept {
                                return *query.QueueKey->GetTarget() == probe->GetTargetIface()->GetLine();
                            });
                        }
                    }
                }
                return storage.FindProbes<TAccessor>(filter);
            }
        }

        const TExpressionId ExpressionId;
        const TTopologyStorage& TopologyStorage;
        TVector<TProbeStorage::TRef> ProbeStorages;
        TPlainLockedBox<TProbe::TRefVector> ResultBox;
    };
}
