#pragma once

#include <infra/netmon/probe.h>
#include <infra/netmon/host_storage.h>

namespace NNetmon {
    namespace {
        class TTreePosition: public TMoveOnly {
        public:
            inline TTreePosition(const TProbe::TSwitchTree& tree)
                : Iterator(tree.Begin())
                , End(tree.End())
            {
            }

            template <class TKey>
            inline TTreePosition(const TProbe::TSwitchTree& tree, const TKey& since)
                : Iterator(tree.LowerBound(since))
                , End(tree.End())
            {
            }

            inline bool IsEnd() const noexcept {
                return Iterator == End;
            }

            inline TProbe* Pop() noexcept {
                Y_ASSERT(!IsEnd());
                auto* probe(Iterator->Get());
                ++Iterator;
                return probe;
            }

            inline bool operator<(const TTreePosition& rhs) const noexcept {
                Y_ASSERT(!rhs.IsEnd() && !IsEnd());
                return (**rhs.Iterator).GetSwitchKey() < (**Iterator).GetSwitchKey();
            }

        private:
            TProbe::TSwitchTree::TConstIterator Iterator;
            TProbe::TSwitchTree::TConstIterator End;
        };

        class TTreeMerger: public TMoveOnly {
        public:
            template <typename... Args>
            inline void Append(const TProbe::TSwitchTree& tree, Args&&... args) noexcept {
                Positions.emplace_back(tree, std::forward<Args>(args)...);
                if (Positions.back().IsEnd()) {
                    Positions.pop_back();
                } else {
                    PushHeap(Positions.begin(), Positions.end());
                }
            }

            inline TProbe* Pop() noexcept {
                Y_ASSERT(!Positions.empty());
                PopHeap(Positions.begin(), Positions.end());
                auto* probe(Positions.back().Pop());
                Y_ASSERT(probe != nullptr);
                if (Positions.back().IsEnd()) {
                    Positions.pop_back();
                } else {
                    PushHeap(Positions.begin(), Positions.end());
                }
                return probe;
            }

            inline bool Empty() const noexcept {
                return Positions.empty();
            }

        private:
            TVector<TTreePosition> Positions;
        };
    }

    // maintain probe index on switch level
    class TProbeIndex: public TNonCopyable {
    public:
        TProbeIndex(const TProbe::TGeneratedTree& generatedTree,
                    const TDuration& window,
                    TCountedHostStorage* seenHosts=nullptr)
            : GeneratedTree(generatedTree)
            , Window(window)
            , LastDeadline(TInstant::Zero())
            , ProbeCount(0)
            , SeenHosts(seenHosts)
        {
        }

        inline void SetLookupTree(TProbeIndex* tree) noexcept {
            LookupTree = tree;
        }

        inline void SetUnlinkTree(TProbeIndex* tree) noexcept {
            UnlinkTree = tree;
        }

        inline std::size_t GetProbeCount() const noexcept {
            std::size_t result = 0;
            auto* index = this;
            while (index != nullptr) {
                result += index->ProbeCount.Val();
                index = index->LookupTree;
            }
            return result;
        }

        inline bool LinkProbe(TProbe& probe) noexcept {
            if (probe.GetGenerated() >= LastDeadline) {
                CurrentTree.Insert(&probe.GetSwitchItem());
                ProbeCount.Inc();
                if (SeenHosts) {
                    SeenHosts->Add(probe.GetSourceIface().GetHost());
                }
                return true;
            } else if (UnlinkTree != nullptr) {
                return UnlinkTree->LinkProbe(probe);
            } else {
                return false;
            }
        }

        std::size_t UnlinkExpiredProbes(const TInstant& now) noexcept {
            std::size_t unlinked = 0;
            auto deadline = now - Window;
            TProbe::TGeneratedTree::TConstIterator it(GeneratedTree.LowerBound(LastDeadline));
            for (; it != GeneratedTree.End(); ++it) {
                TProbe& probe(**it);
                if (probe.GetGenerated() >= deadline) {
                    break;
                }
                if (UnlinkProbe(probe)) {
                    ++unlinked;
                    if (UnlinkTree != nullptr) {
                        UnlinkTree->LinkProbe(probe);
                    }
                }
            }
            LastDeadline = deadline;
            return unlinked;
        }

        template <class TKey, class TFunc>
        void FindProbes(const TKey& key, TFunc&& func) const {
            auto merger(CreateMerger(key));
            while (!merger.Empty()) {
                auto* probe(merger.Pop());
                if (std::is_same<TKey, TSwitchPairKey>::value) {
                    if (!key.Contains(*probe)) {
                        break;
                    }
                } else {
                    if (!key.Contains(*probe)) {
                        continue;
                    } else if (!key.SameTarget(*probe->GetTargetIface())) {
                        break;
                    }
                }
                func(probe);
            }
        }

        template <class TFunc>
        void ForEachProbe(TFunc&& func) const {
            TMaybe<TSwitchPairKey> currentKey;
            TMaybe<TTreeMerger> merger(CreateMerger());
            while (!merger->Empty()) {
                auto* probe(merger->Pop());
                if (currentKey.Defined() && currentKey->Contains(*probe)) {
                    func(probe);
                } else {
                    currentKey.ConstructInPlace(*probe);
                    func(nullptr);
                    // iterators may be invalidated after call, restart search
                    merger.ConstructInPlace(CreateMerger(*currentKey));
                }
            }
        }

    private:
        inline bool UnlinkProbe(TProbe& probe) noexcept {
            auto& item(probe.GetSwitchItem());
            if (item.Linked()) {
                item.UnLink();
                Y_ASSERT(ProbeCount.Val() > 0);
                ProbeCount.Dec();
                if (SeenHosts) {
                    SeenHosts->Remove(probe.GetSourceIface().GetHost());
                }
                return true;
            }
            return false;
        }

        template <typename... Args>
        inline TTreeMerger CreateMerger(Args&&... args) const noexcept {
            TTreeMerger merger;
            auto* index = this;
            while (index != nullptr) {
                merger.Append(index->CurrentTree, std::forward<Args>(args)...);
                index = index->LookupTree;
            }
            return merger;
        }

        const TProbe::TGeneratedTree& GeneratedTree;

        const TDuration Window;
        TInstant LastDeadline;
        TAtomicCounter ProbeCount;

        TCountedHostStorage* SeenHosts;

        TProbe::TSwitchTree CurrentTree;
        // used to find probes across multiple indices
        // for example, switch index should also look for probes across switch and line indices
        TProbeIndex* LookupTree = nullptr;
        // used to move probes from one tree to another on expire
        // for example, datacenter index move probes into line index
        TProbeIndex* UnlinkTree = nullptr;
    };
}
