#include <infra/netmon/probe_schedule_maintainer.h>
#include <infra/netmon/common_ut.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/common_ut.h>
#include <infra/netmon/topology/settings.h>

#include <library/cpp/testing/unittest/registar.h>

#define STRINGIFY(x) #x
#define TO_STRING(x) STRINGIFY(x)
#define CHECK_SCHEDULE_COMMENT "\nCheckSchedule called at " __FILE__ ":" TO_STRING(__LINE__)

class TDummyHostsMaintainer : public IHostsMaintainer {
public:
    TTopologyStorage::THostSetBox::TConstValueRef GetHosts() const override {
        return MakeAtomicShared<TTopologyStorage::THostSet>();
    }
};

class TProbeScheduleMaintainerTest: public TTestBase {
    UNIT_TEST_SUITE(TProbeScheduleMaintainerTest);
    UNIT_TEST(TestCrossDcFullMeshSchedule)
    UNIT_TEST(TestCrossDcFullMeshWithBadHostSchedule)
    UNIT_TEST_SUITE_END();

public:
    TProbeScheduleMaintainerTest()
    {
    }

private:
    inline void CheckSchedule(TProbeScheduleMaintainer& maintainer,
                              const TVector<TTopology::THostRef>& hosts,
                              const TStringBuf& comment,
                              const TTopologyStorage::THostSet& mutedHosts = {}) {
        Sleep(TDuration::MilliSeconds(1));
        for (const auto& host : hosts) {
            maintainer.AddInterestedHost(host);
        }
        maintainer.SpinAndWait().Wait();

        // first iteration ignored our interested hosts due to InterestedHostsChangeThreshold
        Sleep(TDuration::MilliSeconds(1));
        for (const auto& host : hosts) {
            maintainer.AddInterestedHost(host);
        }
        maintainer.SpinAndWait().Wait();

        auto schedule = maintainer.GetSchedule();

        UNIT_ASSERT_C(schedule, comment);
        for (const auto& source : hosts) {
            bool muted = mutedHosts.contains(source);
            UNIT_ASSERT_C(schedule->contains(source) ^ muted, comment);
            if (muted)
                continue;
            for (const auto& target : hosts) {
                if (target != source) {
                    UNIT_ASSERT_C(bool(FindPtr(schedule->at(source).IntraDc, target)) ^ mutedHosts.contains(target), comment);
                    UNIT_ASSERT_C(schedule->at(source).CrossDc.empty(), comment);
                }
            }
        }
    }

    inline void TestCrossDcFullMeshSchedule() {
        TSettings::Get()->SetProbeRescheduleInterval(TDuration::MicroSeconds(1));
        TSettings::Get()->SetCrossDcProbeScheduleFullMesh(true);
        TSettings::Get()->SetProbeScheduleDcs({"myt"});
        // disable normal cross-dc probes
        TSettings::Get()->SetScheduledCrossDcProbesBetweenTwoPods(0);

        TVector<TTopology::THostRef> hosts = {
            FindHost("sas1-6666.search.yandex.net"),
            FindHost("sas3-6666.search.yandex.net"),
            FindHost("vla1-1234.search.yandex.net"),
            FindHost("vla2-1234.search.yandex.net"),
            FindHost("man1-1234.search.yandex.net"),
            FindHost("man2-1234.search.yandex.net")
        };
        THashSet<TString> knownQueues; // reduce known queues to shorten test running time
        for (const auto& host : hosts) {
            knownQueues.emplace(host->GetLine().GetName());
        }
        TTopologySettings::Get()->SetKnownQueues(knownQueues);

        auto& topologyStorage = TGlobalTopology::GetTopologyStorage();
        auto dummyWalleUpdater = TDummyHostsMaintainer();
        auto dummyRtUpdater = TDummyHostsMaintainer();
        TProbeScheduleMaintainer maintainer(topologyStorage, dummyWalleUpdater, dummyRtUpdater);
        CheckSchedule(maintainer, hosts, CHECK_SCHEDULE_COMMENT);
    }

    inline void TestCrossDcFullMeshWithBadHostSchedule() {
        auto muteDuration = TDuration::MilliSeconds(10);

        TSettings::Get()->SetProbeRescheduleInterval(TDuration::MicroSeconds(1));
        TSettings::Get()->SetLinkPollerHostMuteDuration(muteDuration);
        TSettings::Get()->SetCrossDcProbeScheduleFullMesh(true);
        TSettings::Get()->SetNocSlaSwitchMapCapacity(10000);
        TSettings::Get()->SetProbeScheduleDcs({"vla"});

        TVector<TTopology::THostRef> hosts = {
            FindHost("sas1-6666.search.yandex.net"),
            FindHost("sas3-6666.search.yandex.net"),
            FindHost("vla1-1234.search.yandex.net"),
            FindHost("vla2-1234.search.yandex.net"),
            FindHost("man1-1234.search.yandex.net"),
            FindHost("man2-1234.search.yandex.net")
        };
        THashSet<TString> knownQueues; // reduce known queues to shorten test running time
        for (const auto& host : hosts) {
            knownQueues.emplace(host->GetLine().GetName());
        }
        TTopologySettings::Get()->SetKnownQueues(knownQueues);

        auto dummyWalleUpdater = TDummyHostsMaintainer();
        auto dummyRtUpdater = TDummyHostsMaintainer();

        const auto& topologyStorage = TGlobalTopology::GetTopologyStorage();

        auto mutedHost = FindHost("vla1-1234.search.yandex.net");
        auto normalHost = FindHost("vla2-1234.search.yandex.net");

        TProbeScheduleMaintainer maintainer(topologyStorage, dummyWalleUpdater, dummyRtUpdater);
        maintainer.AddLinkPollerStats(mutedHost, 1000, 100);
        maintainer.AddLinkPollerStats(mutedHost, 900, 0);
        maintainer.AddLinkPollerStats(normalHost, 1901, 99);

        auto schedule = maintainer.GetSchedule();
        CheckSchedule(maintainer, hosts, CHECK_SCHEDULE_COMMENT, {mutedHost});
        Sleep(muteDuration);

        CheckSchedule(maintainer, hosts, CHECK_SCHEDULE_COMMENT);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TProbeScheduleMaintainerTest);
