#pragma once

#include <infra/netmon/topology/topology_storage.h>

#include <util/digest/multi.h>

namespace NNetmon {
    class IProbeScheduler {
    public:
        struct THostTargets {
            TVector<TTopology::THostRef> IntraDc, CrossDc, IntraDcSparse, Ipv4IntraDc;
        };
        using TProbeSchedule = THashMap<TTopology::THostRef, THostTargets>;

        virtual ~IProbeScheduler() = default;

        virtual TProbeSchedule Schedule(const TTopologyStorage::THostSet& interestedHosts,
                                        const TTopologyStorage::THostSet& deadHosts,
                                        const TTopologyStorage::THostSet& crossDcTargetHosts = {},
                                        const TTopologyStorage::THostSet& mutedHosts = {}) const = 0;
    };

    class TUniformSwitchProbeScheduler: public IProbeScheduler, public TNonCopyable {
    public:
        // TSwitchPair(sw1, sw2) == TSwitchPair(sw2, sw1)
        struct TSwitchPair {
            inline TSwitchPair(TTopology::TSwitchRef first, TTopology::TSwitchRef second)
                : First(first->GetReducedId() < second->GetReducedId() ? first : second)
                , Second(First == first ? second : first)
            {
            }

            inline TSwitchPair(const TSwitch& first, const TSwitch& second)
                : TSwitchPair(TTopology::TSwitchRef(first), TTopology::TSwitchRef(second))
            {
            }

            inline size_t GetHash() const noexcept {
                return MultiHash(First->GetHash(), Second->GetHash());
            }

            inline bool operator==(const TSwitchPair& rhs) const noexcept {
                return First == rhs.First && Second == rhs.Second;
            }

            TTopology::TSwitchRef First;
            TTopology::TSwitchRef Second;
        };

        using TSwitchFilter = std::function<bool(const TSwitch&)>;

        TUniformSwitchProbeScheduler(const TTopologyStorage& topologyStorage,
                                     TSwitchFilter filter,
                                     THashMap<TString, std::size_t> probesBetweenTwoSwitches,
                                     std::size_t crossDcProbesBetweenTwoPods);
        ~TUniformSwitchProbeScheduler();

        TProbeSchedule Schedule(const TTopologyStorage::THostSet& interestedHosts,
                                const TTopologyStorage::THostSet& deadHosts,
                                const TTopologyStorage::THostSet& crossDcTargetHosts = {},
                                const TTopologyStorage::THostSet& mutedHosts = {}) const override;

    private:
        class TImpl;
        THolder<TImpl> Impl;
    };
}

template <>
class THash<NNetmon::TUniformSwitchProbeScheduler::TSwitchPair> {
public:
    size_t operator()(const NNetmon::TUniformSwitchProbeScheduler::TSwitchPair& pair) const {
        return pair.GetHash();
    }
};
