#include <infra/netmon/probe_slice.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    namespace {
        template <class T, class TProcessor, class TExecutor>
        class TTaskRunner : public TScheduledTask {
        public:
            TTaskRunner(const TDuration& interval, TProbeSlice& slice)
                : TScheduledTask(interval)
                , Slice(slice)
            {
            }

            TThreadPool::TFuture Run() override {
                return TExecutor::Get()->Add([this]() {
                    TSimpleTimer timer;
                    Processor(Slice);
                    DEBUG_LOG << Slice << " processing with " << TypeName<TProcessor>()
                              << " took " << timer.Get() << Endl;
                });
            }

        private:
            TProbeSlice& Slice;
            TProcessor Processor;
        };

        class TSwitchExecutor: public TCustomThreadExecutor<TSwitchExecutor> {
        public:
            TSwitchExecutor()
                : TCustomThreadExecutor("SwitchSliceProcessor")
            {
            }
        };
        class TSwitchTask: public TTaskRunner<TSwitchTask, TProbeSlice::TSwitchProcessor, TSwitchExecutor> {
            using TTaskRunner::TTaskRunner;
        };

        class TLineExecutor: public TCustomThreadExecutor<TLineExecutor> {
        public:
            TLineExecutor()
                : TCustomThreadExecutor("LineSliceProcessor")
            {
            }
        };
        class TLineTask: public TTaskRunner<TLineTask, TProbeSlice::TLineProcessor, TLineExecutor> {
            using TTaskRunner::TTaskRunner;
        };

        class TDatacenterExecutor: public TCustomThreadExecutor<TDatacenterExecutor> {
        public:
            TDatacenterExecutor()
                : TCustomThreadExecutor("DcSliceProcessor")
            {
            }
        };
        class TDatacenterTask: public TTaskRunner<TDatacenterTask, TProbeSlice::TDatacenterProcessor, TDatacenterExecutor> {
            using TTaskRunner::TTaskRunner;
        };
    }

    TProbeSlice::TProbeSlice(const TProbeSliceKey &key,
                             const TTopologyStorage& topologyStorage,
                             const IHostsMaintainer& seenHostsUpdater,
                             const IHostsMaintainer& terminatedHostsMaintainer,
                             const IHostsMaintainer &walleUpdater)
        : Key(key)
        , TopologyStorage(topologyStorage)
        , SeenHostsUpdater(seenHostsUpdater)
        , TerminatedHostsMaintainer(terminatedHostsMaintainer)
        , WalleUpdater(walleUpdater)
        , ProbeStorage(TProbeStorage::Make(Key, TopologyStorage))
        , SwitchState(Key)
        , QueueState(Key)
        , DatacenterState(Key)
        , SwitchTask(MakeHolder<TSwitchTask>(TSettings::Get()->GetSwitchAggregationInterval(), *this))
        , QueueTask(MakeHolder<TLineTask>(TSettings::Get()->GetLineAggregationInterval(), *this))
        , DatacenterTask(MakeHolder<TDatacenterTask>(TSettings::Get()->GetDcAggregationInterval(), *this))
        , SwitchTaskGuard(SwitchTask->Schedule())
        , QueueTaskGuard(QueueTask->Schedule())
        , DatacenterTaskGuard(DatacenterTask->Schedule())
    {
    }

    void TProbeSlice::TSwitchProcessor::operator()(TProbeSlice& slice) {
        TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::PreAggregationSwitchTime};

        TSwitchIndexMap::TRef switchIndexMap(
            slice.GetProbeStorage()->CreateSwitchIndexMap<TProbeStorage::TSwitchIndexAccessor>(
                slice.GetSeenHosts(), slice.GetTerminatedHosts(), slice.GetDeadHosts()));

        slice.SwitchState.Set(switchIndexMap);
    }

    void TProbeSlice::TLineProcessor::operator()(TProbeSlice& slice) {
        TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::PreAggregationQueueTime};

        auto selector(slice.GetTopologySelector());
        TSwitchIndexMap::TRef switchIndexMap(
            slice.GetProbeStorage()->CreateSwitchIndexMap<TProbeStorage::TLineIndexAccessor>(
                slice.GetSeenHosts(), slice.GetTerminatedHosts(), slice.GetDeadHosts()));

        TLineIndexMap::TRef queueIndexMap(
            MakeAtomicShared<TLineIndexMap>(
                *selector, *switchIndexMap));

        slice.QueueState.Set(queueIndexMap);
    }

    void TProbeSlice::TDatacenterProcessor::operator()(TProbeSlice& slice) {
        TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::PreAggregationDcTime};

        auto selector(slice.GetTopologySelector());
        TSwitchIndexMap::TRef switchIndexMap(
            slice.GetProbeStorage()->CreateSwitchIndexMap<TProbeStorage::TDatacenterIndexAccessor>(
                slice.GetSeenHosts(), slice.GetTerminatedHosts(), slice.GetDeadHosts()));

        TDatacenterIndexMap::TRef datacenterIndexMap(
            MakeAtomicShared<TDatacenterIndexMap>(
                *selector, *switchIndexMap));

        slice.DatacenterState.Set(datacenterIndexMap);
    }

    void TProbeSlice::InsertProbes(TVector<TProbe::TRef>&& currentProbes, const TInstant& now) {
        auto stats = ProbeStorage->InsertProbes(std::move(currentProbes), now);
        INFO_LOG << ProbeStorage->GetSwitchProbeCount() << " switch level probes, "
                 << ProbeStorage->GetLineProbeCount() << " queue level probes, "
                 << ProbeStorage->GetDatacenterProbeCount() << " datacenter level probes, "
                 << stats.Created << " added, "
                 << stats.Duplicates << " duplicates, "
                 << stats.Unlinked << " unlinked, "
                 << stats.Deleted << " deleted from " << *this
                 << ", insertion took " << stats.InsertTime
                 << ", deletion took " << stats.DeleteTime
                 << Endl;
    }

    void TProbeSlice::Out(IOutputStream& stream) const {
        stream << "TProbeSlice(key=" << GetKey() << ")";
    }
}

template <>
void Out<NNetmon::TProbeSlice>(IOutputStream& stream,
                               TTypeTraits<NNetmon::TProbeSlice>::TFuncParam slice) {
    slice.Out(stream);
}
