#pragma once

#include <infra/netmon/probe.h>
#include <infra/netmon/probe_storage.h>
#include <infra/netmon/host_storage.h>

#include <library/cpp/containers/intrusive_rb_tree/rb_tree.h>

namespace NNetmon {
    namespace {
        struct TProbeSliceCompare {
            template <class T>
            static inline bool Compare(const T& lhs, const T& rhs) {
                return lhs < rhs;
            }

            template <class T>
            static inline bool Compare(const typename T::TKey& lhs, const T& rhs) {
                return lhs < rhs.GetKey();
            }

            template <class T>
            static inline bool Compare(const T& lhs, const typename T::TKey& rhs) {
                return lhs.GetKey() < rhs;
            }

            template <class T>
            static inline bool Compare(const typename T::TSection& lhs, const T& rhs) {
                return lhs < rhs.GetKey().GetSectionKey();
            }

            template <class T>
            static inline bool Compare(const T& lhs, const typename T::TSection& rhs) {
                return lhs.GetKey().GetSectionKey() < rhs;
            }
        };
    }

    class TProbeSlice: public TRbTreeItem<TProbeSlice, TProbeSliceCompare>, public TNonCopyable {
    public:
        using TKey = TProbeSliceKey;
        using TSection = TProbeSectionKey;
        using TTree = TRbTree<TProbeSlice, TProbeSliceCompare>;
        using TRefVector = TVector<const TProbeSlice*>;

        struct TSwitchProcessor {
            void operator()(TProbeSlice& slice);
        };

        struct TLineProcessor {
            void operator()(TProbeSlice& slice);
        };

        struct TDatacenterProcessor {
            void operator()(TProbeSlice& slice);
        };

    private:
        template <class TIndexMap>
        struct TState {
            TState(const TKey& sliceKey)
                : IndexMap(MakeAtomicShared<TIndexMap>(sliceKey))
                , UpdateTime(TInstant::Zero())
            {
            }

            const typename TIndexMap::TRef Get() const {
                auto guard = TGuard<TAdaptiveLock>(Lock);
                return IndexMap;
            }

            void Set(const typename TIndexMap::TRef newIndexMap) {
                auto guard = TGuard<TAdaptiveLock>(Lock);
                IndexMap = newIndexMap;
                UpdateTime = TInstant::Now();
            }

            typename TIndexMap::TRef IndexMap;
            TInstant UpdateTime;

            mutable TAdaptiveLock Lock;
        };

    public:
        TProbeSlice(const TProbeSliceKey& key,
                    const TTopologyStorage& topologyStorage,
                    const IHostsMaintainer& seenHostsUpdater,
                    const IHostsMaintainer& terminatedHostsMaintainer,
                    const IHostsMaintainer& walleUpdater);

        const TKey& GetKey() const {
            return Key;
        }

        bool SameSection(const TSection& section) const {
            return (
                Key.GetNetwork() == section.GetNetwork()
                && Key.GetProtocol() == section.GetProtocol()
            );
        }

        TTopologySelector::TBox::TConstValueRef GetTopologySelector() const {
            return TopologyStorage.GetTopologySelector();
        }
        TTopologyStorage::THostSetBox::TConstValueRef GetSeenHosts() const {
            return SeenHostsUpdater.GetHosts();
        }
        TTopologyStorage::THostSetBox::TConstValueRef GetTerminatedHosts() const {
            return TerminatedHostsMaintainer.GetHosts();
        }
        TTopologyStorage::THostSetBox::TConstValueRef GetDeadHosts() const {
            return WalleUpdater.GetHosts();
        }
        TProbeStorage::TRef GetProbeStorage() const {
            return ProbeStorage;
        }

        TSwitchIndexMap::TRef GetSwitchLevelIndex() const {
            return SwitchState.Get();
        }
        TLineIndexMap::TRef GetLineLevelIndex() const {
            return QueueState.Get();
        }
        TDatacenterIndexMap::TRef GetDatacenterLevelIndex() const {
            return DatacenterState.Get();
        }

        TThreadPool::TFuture WaitSwitchTask() noexcept {
            return SwitchTask->SpinAndWait();
        }
        TThreadPool::TFuture WaitQueueTask() noexcept {
            return QueueTask->SpinAndWait();
        }
        TThreadPool::TFuture WaitDatacenterTask() noexcept {
            return DatacenterTask->SpinAndWait();
        }

        void InsertProbes(TVector<TProbe::TRef>&& currentProbes, const TInstant& now);

        void Out(IOutputStream& stream) const;

        inline bool operator<(const TProbeSlice& rhs) const {
            return Key < rhs.Key;
        }

    private:
        const TProbeSliceKey Key;
        const TTopologyStorage& TopologyStorage;
        const IHostsMaintainer& SeenHostsUpdater;
        const IHostsMaintainer& TerminatedHostsMaintainer;
        const IHostsMaintainer& WalleUpdater;

        TProbeStorage::TRef ProbeStorage;

        TState<TSwitchIndexMap> SwitchState;
        TState<TLineIndexMap> QueueState;
        TState<TDatacenterIndexMap> DatacenterState;

        THolder<TScheduledTask> SwitchTask;
        THolder<TScheduledTask> QueueTask;
        THolder<TScheduledTask> DatacenterTask;

        TScheduledTask::TTaskGuard SwitchTaskGuard;
        TScheduledTask::TTaskGuard QueueTaskGuard;
        TScheduledTask::TTaskGuard DatacenterTaskGuard;
    };
}
