#include <infra/netmon/probe_slice_dumper.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/history_placement.h>
#include <infra/netmon/library/api_client_helpers.h>

namespace NNetmon {
    namespace {
        template <class T, class TIndex>
        class TIndexVisitor : public TNonCopyable {
        public:
            inline TIndexVisitor(T* parent, const TInstant& timestamp)
                : Parent(parent)
                , PrevDumpTime(timestamp)
                , Dumped(false)
            {
            }

            inline std::tuple<bool, TInstant> Visit() {
                const auto keys(Parent->GetKeys());
                for (const auto& key : keys) {
                    const auto index(Parent->GetIndex(key));
                    if (index) {
                        VisitIndex(key, *index);
                    }
                }

                if (Timestamp == PrevDumpTime) {
                    return {false, TInstant::Zero()};
                } else {
                    return {Dumped, Timestamp};
                }
            }

        private:
            void VisitIndex(const TProbeAggregatorKey& key, const TIndex& index) {
                for (auto it(index.GetTree().Begin()); it != index.GetTree().End(); ++it) {
                    Parent->VisitState(key, index, *it);
                    Dumped = true;
                }
                Parent->OnIndexVisited(key, index);
                Timestamp = Max(Timestamp, index.GetGenerated());
            }

            T* Parent;
            const TInstant PrevDumpTime;
            TInstant Timestamp;
            bool Dumped;
        };

        template <class T, class TIndex>
        class TEmbeddedWriter : public TNonCopyable {
        public:
            using TSelf = TEmbeddedWriter<T, TIndex>;
            using TState = typename TIndex::TState;
            using TFlatStateVector = std::vector<flatbuffers::Offset<typename TState::TFlat>>;
            using TParentKey = typename TState::TKey::TParent;

            friend TIndexVisitor<TSelf, TIndex>;

            inline TEmbeddedWriter(const T* parent, NNetmon::THistoryChunkBuilder& builder, const TInstant& timestamp)
                : Parent(parent)
                , Builder(builder)
                , PrevDumpTime(timestamp)
            {
            }

            std::tuple<bool, TInstant> Visit() {
                TIndexVisitor<TSelf, TIndex> visitor(this, PrevDumpTime);
                return visitor.Visit();
            }

        private:
            inline TProbeAggregatorKeySet GetKeys() const {
                return Parent->GetKeys();
            }

            inline typename TIndex::TRef GetIndex(const TProbeAggregatorKey& key) const {
                return Parent->GetIndex(key);
            }

            void VisitState(const TProbeAggregatorKey& key,
                            const TIndex& index,
                            const TState& state)
            {
                const auto pairKey(state.GetKey().GetParent());

                if (!CurrentPair.Defined() || pairKey != *CurrentPair) {
                    Dump(key, index);
                    CurrentPair.ConstructInPlace(pairKey);
                }

                FlatStates.emplace_back(state.ToProto(FlatBuilder));
            }

            void OnIndexVisited(const TProbeAggregatorKey& key, const TIndex& index)
            {
                Dump(key, index);
                CurrentPair.Clear();
            }

            void Dump(const TProbeAggregatorKey& key, const TIndex& index) {
                if (!FlatStates.empty()) {
                    Parent->ToProto(FlatBuilder, key, index, FlatStates);
                    const auto dumpedKey(ToHistoryKey(key, *CurrentPair).AsTuple());
                    Builder.Add(dumpedKey, ToStringBuf(FlatBuilder));
                    FlatStates.clear();
                    FlatBuilder.Clear();
                }
            }

            const T* Parent;
            NNetmon::THistoryChunkBuilder& Builder;
            const TInstant PrevDumpTime;
            TMaybe<TParentKey> CurrentPair;
            flatbuffers::FlatBufferBuilder FlatBuilder;
            TFlatStateVector FlatStates;
        };

        template <class T, class TIndex>
        class TRootWriter : public TNonCopyable {
        public:
            using TSelf = TRootWriter<T, TIndex>;
            using TState = typename TIndex::TState;
            using TFlatStateVector = std::vector<flatbuffers::Offset<typename TState::TFlat>>;

            friend TIndexVisitor<TSelf, TIndex>;

            inline TRootWriter(const T* parent, NNetmon::THistoryChunkBuilder& builder, const TInstant& timestamp)
                : Parent(parent)
                , Builder(builder)
                , PrevDumpTime(timestamp)
            {
            }

            std::tuple<bool, TInstant> Visit() {
                TIndexVisitor<TSelf, TIndex> visitor(this, PrevDumpTime);
                return visitor.Visit();
            }

        private:
            inline TProbeAggregatorKeySet GetKeys() const {
                return Parent->GetKeys();
            }

            inline typename TIndex::TRef GetIndex(const TProbeAggregatorKey& key) const {
                return Parent->GetIndex(key);
            }

            void VisitState(const TProbeAggregatorKey&, const TIndex&, const TState& state)
            {
                FlatStates.emplace_back(state.ToProto(FlatBuilder));
            }

            void OnIndexVisited(const TProbeAggregatorKey& key, const TIndex& index)
            {
                if (!FlatStates.empty()) {
                    Parent->ToProto(FlatBuilder, key, index, FlatStates);
                    const auto dumpedKey(ToHistoryKey(key).AsTuple());
                    Builder.Add(dumpedKey, ToStringBuf(FlatBuilder));
                    FlatStates.clear();
                    FlatBuilder.Clear();
                }
            }

        private:
            const T* Parent;
            NNetmon::THistoryChunkBuilder& Builder;
            const TInstant PrevDumpTime;
            flatbuffers::FlatBufferBuilder FlatBuilder;
            TFlatStateVector FlatStates;
        };

        class TSwitchConfig : public TNonCopyable {
        public:
            using THistoryKey = THistoryLinePairKey;
            using TWriter = TEmbeddedWriter<TSwitchConfig, TSwitchPairIndex>;

            inline TSwitchConfig(const TTopologyStorage& topologyStorage, const IProbeSliceMerger& merger)
                : TopologyStorage(topologyStorage)
                , Merger(merger)
            {
            }

            std::tuple<bool, TInstant> FillBuilder(NNetmon::THistoryChunkBuilder& builder, const TInstant& timestamp) const {
                TWriter writer(this, builder, timestamp);
                return writer.Visit();
            }

            bool HasAnythingToDump() const {
                return true;
            }

            inline TStringBuf GetCodec() const {
                return TStringBuf("zstd08_1");
            }

            inline ENetmonSignals ReadMetric() const {
                return ENetmonSignals::StateDumperSwitchReadTime;
            }

            inline ENetmonSignals DumpMetric() const {
                return ENetmonSignals::StateDumperSwitchWriteTime;
            }

            inline TDuration GetChunkDuration() const {
                return TSettings::Get()->GetSwitchAggregationInterval();
            }

            inline TDuration GetTimeToLive() const {
                return TSettings::Get()->GetSwitchHistoryTimeToLive();
            }

            inline TDuration GetDumpInterval() const {
                return TSettings::Get()->GetSwitchAggregationInterval();
            }

            inline std::size_t GetBlobCacheSize() const {
                return 150;
            }

            inline TFsPath GetRoot() const {
                return TSettings::Get()->GetHistoryStatePath().Child("switch");
            }

            inline TProbeAggregatorKeySet GetKeys() const {
                return Merger.GetSwitchLevelKeys();
            }

            inline TSwitchPairIndex::TRef GetIndex(const TProbeAggregatorKey& key) const {
                return Merger.GetSwitchLevelIndex(key);
            }

            inline TInstant RoundToInterval(const TInstant& ts) const {
                return RoundInstant(ts, GetChunkDuration());
            }

            inline void ToProto(flatbuffers::FlatBufferBuilder& builder,
                                const TProbeAggregatorKey& key,
                                const TSwitchPairIndex& index,
                                const TWriter::TFlatStateVector& states) const {
                builder.Finish(NAggregation::CreateTSwitchPairIndex(
                    builder,
                    &key.ToProto(),
                    index.GetPairCount(),
                    builder.CreateVector(states),
                    0,
                    index.GetGenerated().MicroSeconds()
                ));
            }

            inline THistoryKey RestoreKey(const THistoryTuple& key) const {
                return THistoryKey(TopologyStorage, key);
            }

            inline TSwitchPairIndex::TRef FromProto(const TStringBuf& payload) const {
                return TSwitchPairIndex::Make(
                    TopologyStorage,
                    *flatbuffers::GetRoot<TSwitchPairIndex::TFlat>(payload.data())
                );
            }

            TSwitchPairIndex::TRef GetRealtime(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
                const auto index(GetIndex(key));
                if (index && RoundToInterval(index->GetGenerated()) == RoundToInterval(requestedTs)) {
                    return index->MakeSubset(pairKey);
                } else {
                    return nullptr;
                }
            }

        private:
            const TTopologyStorage& TopologyStorage;
            const IProbeSliceMerger& Merger;
        };

        class TLineConfig : public TNonCopyable {
        public:
            using THistoryKey = THistoryDatacenterPairKey;
            using TWriter = TEmbeddedWriter<TLineConfig, TLinePairIndex>;

            inline TLineConfig(const TTopologyStorage& topologyStorage, const IProbeSliceMerger& merger)
                : TopologyStorage(topologyStorage)
                , Merger(merger)
            {
            }

            std::tuple<bool, TInstant> FillBuilder(NNetmon::THistoryChunkBuilder& builder, const TInstant& timestamp) const {
                TWriter writer(this, builder, timestamp);
                return writer.Visit();
            }

            bool HasAnythingToDump() const {
                return true;
            }

            inline TStringBuf GetCodec() const {
                return TStringBuf("zstd08_1");
            }

            inline ENetmonSignals ReadMetric() const {
                return ENetmonSignals::StateDumperQueueReadTime;
            }

            inline ENetmonSignals DumpMetric() const {
                return ENetmonSignals::StateDumperQueueWriteTime;
            }

            inline TDuration GetChunkDuration() const {
                return TSettings::Get()->GetLineAggregationInterval();
            }

            inline TDuration GetTimeToLive() const {
                return TSettings::Get()->GetQueueHistoryTimeToLive();
            }

            inline TDuration GetDumpInterval() const {
                return TSettings::Get()->GetLineAggregationInterval();
            }

            inline std::size_t GetBlobCacheSize() const {
                return 500;
            }

            inline TFsPath GetRoot() const {
                return TSettings::Get()->GetHistoryStatePath().Child("line");
            }

            inline TProbeAggregatorKeySet GetKeys() const {
                return Merger.GetLineLevelKeys();
            }

            inline TLinePairIndex::TRef GetIndex(const TProbeAggregatorKey& key) const {
                return Merger.GetLineLevelIndex(key);
            }

            inline TInstant RoundToInterval(const TInstant& ts) const {
                return RoundInstant(ts, GetChunkDuration());
            }

            inline void ToProto(flatbuffers::FlatBufferBuilder& builder,
                                const TProbeAggregatorKey& key,
                                const TLinePairIndex& index,
                                const TWriter::TFlatStateVector& states) const {
                builder.Finish(NAggregation::CreateTLinePairIndex(
                    builder,
                    &key.ToProto(),
                    index.GetPairCount(),
                    builder.CreateVector(states),
                    0,
                    index.GetGenerated().MicroSeconds()
                ));
            }

            inline THistoryKey RestoreKey(const THistoryTuple& key) const {
                return THistoryKey(TopologyStorage, key);
            }

            inline TLinePairIndex::TRef FromProto(const TStringBuf& payload) const {
                return TLinePairIndex::Make(
                    TopologyStorage,
                    *flatbuffers::GetRoot<TLinePairIndex::TFlat>(payload.data())
                );
            }

            TLinePairIndex::TRef GetRealtime(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
                const auto index(GetIndex(key));
                if (index && RoundToInterval(index->GetGenerated()) == RoundToInterval(requestedTs)) {
                    if (!pairKey.IsValid()) {
                        return index;
                    } else {
                        return index->MakeSubset(pairKey);
                    }
                } else {
                    return nullptr;
                }
            }

        private:
            const TTopologyStorage& TopologyStorage;
            const IProbeSliceMerger& Merger;
        };

        class TDatacenterConfig : public TNonCopyable {
        public:
            using THistoryKey = THistoryRootKey;
            using TWriter = TRootWriter<TDatacenterConfig, TDatacenterPairIndex>;

            inline TDatacenterConfig(const TTopologyStorage& topologyStorage, const IProbeSliceMerger& merger)
                : TopologyStorage(topologyStorage)
                , Merger(merger)
            {
            }

            std::tuple<bool, TInstant> FillBuilder(NNetmon::THistoryChunkBuilder& builder, const TInstant& timestamp) const {
                TWriter writer(this, builder, timestamp);
                return writer.Visit();
            }

            bool HasAnythingToDump() const {
                return true;
            }

            inline TStringBuf GetCodec() const {
                return TStringBuf("zstd08_1");
            }

            inline ENetmonSignals ReadMetric() const {
                return ENetmonSignals::StateDumperDcReadTime;
            }

            inline ENetmonSignals DumpMetric() const {
                return ENetmonSignals::StateDumperDcWriteTime;
            }

            inline TDuration GetChunkDuration() const {
                return TSettings::Get()->GetDcAggregationInterval();
            }

            inline TDuration GetTimeToLive() const {
                return TSettings::Get()->GetDcHistoryTimeToLive();
            }

            inline TDuration GetDumpInterval() const {
                return TSettings::Get()->GetDcAggregationInterval();
            }

            inline std::size_t GetBlobCacheSize() const {
                return 1000;
            }

            inline TFsPath GetRoot() const {
                return TSettings::Get()->GetHistoryStatePath().Child("dc");
            }

            inline TProbeAggregatorKeySet GetKeys() const {
                return Merger.GetDatacenterLevelKeys();
            }

            inline TDatacenterPairIndex::TRef GetIndex(const TProbeAggregatorKey& key) const {
                return Merger.GetDatacenterLevelIndex(key);
            }

            inline TInstant RoundToInterval(const TInstant& ts) const {
                return RoundInstant(ts, GetChunkDuration());
            }

            inline void ToProto(flatbuffers::FlatBufferBuilder& builder,
                                const TProbeAggregatorKey& key,
                                const TDatacenterPairIndex& index,
                                const TWriter::TFlatStateVector& states) const {
                builder.Finish(NAggregation::CreateTDatacenterPairIndex(
                    builder,
                    &key.ToProto(),
                    index.GetPairCount(),
                    builder.CreateVector(states),
                    0,
                    index.GetGenerated().MicroSeconds()
                ));
            }

            inline THistoryKey RestoreKey(const THistoryTuple& key) const {
                return THistoryKey(TopologyStorage, key);
            }

            inline TDatacenterPairIndex::TRef FromProto(const TStringBuf& payload) const {
                return TDatacenterPairIndex::Make(
                    TopologyStorage,
                    *flatbuffers::GetRoot<TDatacenterPairIndex::TFlat>(payload.data())
                );
            }

            TDatacenterPairIndex::TRef GetRealtime(const TInstant& requestedTs, const TProbeAggregatorKey& key) const {
                const auto index(GetIndex(key));
                if (index && RoundToInterval(index->GetGenerated()) == RoundToInterval(requestedTs)) {
                    return index;
                } else {
                    return nullptr;
                }
            }

        private:
            const TTopologyStorage& TopologyStorage;
            const IProbeSliceMerger& Merger;
        };

        using TSwitchMaintainer = THistoryMaintainer<TSwitchConfig>;
        using TLineMaintainer = THistoryMaintainer<TLineConfig>;
        using TDatacenterMaintainer = THistoryMaintainer<TDatacenterConfig>;

        template <class TMaintainer, typename... Args>
        inline auto RealtimeAwareRead(const TMaintainer& maintainer, const TInstant& requestedTs, const Args&... args) {
            TBuffer buffer;
            TStringBuf payload;
            const auto key(ToHistoryKey(args...).AsTuple());
            const auto index(maintainer.GetConfig().GetRealtime(requestedTs, args...));
            if (index) {
                return index;
            } else if (maintainer.Read(key, requestedTs, payload, buffer)) {
                return maintainer.GetConfig().FromProto(payload);
            } else {
                // this may be nullptr if type deduction would be smarter
                return index;
            }
        }

        /* Explicit template specialization for queues.
           It's still safe for queue history to return full mesh.
           However reading full mesh for switch may impose severe performance
           penalty on the server and further problems */

        template <typename... Args>
        inline auto RealtimeAwareRead(const TLineMaintainer& maintainer, const TInstant& requestedTs, const Args&... args) {
            const auto pairKey(ToHistoryKey(args...).GetPairKey());
            const auto index(maintainer.GetConfig().GetRealtime(requestedTs, args...));
            if (index) {
                return index;
            } else if (!pairKey.IsValid()) {
                const auto aggrKey(ToHistoryKey(args...).GetAggregatorKey());
                auto wildcardIndex = TLinePairIndex::Make(aggrKey, requestedTs);

                maintainer.Iterate(
                    requestedTs,
                    [&wildcardIndex, &maintainer, &aggrKey]
                    (THistoryTuple key, TStringBuf payload) {
                        if (std::get<0>(key) == aggrKey.GetExpressionId() &&
                            std::get<1>(key) == aggrKey.GetNetwork() &&
                            std::get<2>(key) == aggrKey.GetProtocol()) {

                            wildcardIndex->MergeFrom(*(maintainer.GetConfig().FromProto(payload)));
                        }
                    }
                );

                return wildcardIndex;
            }

            TBuffer buffer;
            TStringBuf payload;
            const auto key(ToHistoryKey(args...).AsTuple());

            if (maintainer.Read(key, requestedTs, payload, buffer)) {
                return maintainer.GetConfig().FromProto(payload);
            } else {
                return index;
            }
        }

        template <class TMaintainer, class F>
        inline void IterateOverData(const TMaintainer& maintainer, const TInstant& requestedTs, F&& func) {
            const auto& config(maintainer.GetConfig());
            maintainer.Iterate(requestedTs, [&func, &config](THistoryTuple key, TStringBuf payload) {
                func(config.RestoreKey(key), config.FromProto(payload));
            });
        }

        template <class TMaintainer, class F>
        inline void IterateOverData(const TMaintainer& maintainer, const TInstant& since, const TInstant& until, F&& func) {
            const auto& config(maintainer.GetConfig());
            auto epoch(config.RoundToInterval(since));
            while (epoch <= config.RoundToInterval(until)) {
                IterateOverData(maintainer, epoch, func);
                epoch += config.GetChunkDuration();
            }
        }
    }

    class TProbeSliceDumper::TImpl {
    public:
        TImpl(const TTopologyStorage& topologyStorage, const IProbeSliceMerger& merger)
            : SwitchConfig(topologyStorage, merger)
            , LineConfig(topologyStorage, merger)
            , DatacenterConfig(topologyStorage, merger)
            , SwitchMaintainer(SwitchConfig)
            , LineMaintainer(LineConfig)
            , DatacenterMaintainer(DatacenterConfig)
        {
        }

        inline bool IsReady() const noexcept {
            return (
                SwitchMaintainer.IsReady()
                && LineMaintainer.IsReady()
                && DatacenterMaintainer.IsReady()
            );
        }

        inline TSwitchPairIndex::TRef Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
            return RealtimeAwareRead(SwitchMaintainer, requestedTs, key, pairKey);
        }

        inline TLinePairIndex::TRef Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
            return RealtimeAwareRead(LineMaintainer, requestedTs, key, pairKey);
        }

        inline TDatacenterPairIndex::TRef Read(const TInstant& requestedTs, const TProbeAggregatorKey& key) const {
            return RealtimeAwareRead(DatacenterMaintainer, requestedTs, key);
        }

        template <class F>
        inline void IterateSwitch(const TInstant& requestedTs, F&& func) const {
            IterateOverData(SwitchMaintainer, requestedTs, std::move(func));
        }

        template <class F>
        inline void IterateLine(const TInstant& requestedTs, F&& func) const {
            IterateOverData(LineMaintainer, requestedTs, std::move(func));
        }

        template <class F>
        inline void IterateDatacenter(const TInstant& requestedTs, F&& func) const {
            IterateOverData(DatacenterMaintainer, requestedTs, std::move(func));
        }

        template <class F>
        inline void IterateSwitch(const TInstant& since, const TInstant& until, F&& func) const {
            IterateOverData(SwitchMaintainer, since, until, std::move(func));
        }

        template <class F>
        inline void IterateLine(const TInstant& since, const TInstant& until, F&& func) const {
            IterateOverData(LineMaintainer, since, until, std::move(func));
        }

        template <class F>
        inline void IterateDatacenter(const TInstant& since, const TInstant& until, F&& func) const {
            IterateOverData(DatacenterMaintainer, since, until, std::move(func));
        }

        TThreadPool::TFuture WaitSwitchTask() {
            return SwitchMaintainer.WaitDumper();
        }

        TThreadPool::TFuture WaitLineTask() {
            return LineMaintainer.WaitDumper();
        }

        TThreadPool::TFuture WaitDatacenterTask() {
            return DatacenterMaintainer.WaitDumper();
        }

    private:

        TSwitchConfig SwitchConfig;
        TLineConfig LineConfig;
        TDatacenterConfig DatacenterConfig;

        TSwitchMaintainer SwitchMaintainer;
        TLineMaintainer LineMaintainer;
        TDatacenterMaintainer DatacenterMaintainer;
    };

    TProbeSliceDumper::TProbeSliceDumper(const TTopologyStorage& topologyStorage,
                                         const IProbeSliceMerger& sliceMerger)
        : Impl(MakeHolder<TImpl>(topologyStorage, sliceMerger))
    {
    }

    TProbeSliceDumper::~TProbeSliceDumper() {
    }

    bool TProbeSliceDumper::IsReady() const noexcept {
        return Impl->IsReady();
    }

    TSwitchPairIndex::TRef TProbeSliceDumper::Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
        return Impl->Read(requestedTs, key, pairKey);
    }

    TLinePairIndex::TRef TProbeSliceDumper::Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
        return Impl->Read(requestedTs, key, pairKey);
    }

    TDatacenterPairIndex::TRef TProbeSliceDumper::Read(const TInstant& requestedTs, const TProbeAggregatorKey& key) const {
        return Impl->Read(requestedTs, key);
    }

    void TProbeSliceDumper::IterateSwitch(const TInstant& requestedTs, TSwitchCallback func) const {
        Impl->IterateSwitch(requestedTs, std::move(func));
    }

    void TProbeSliceDumper::IterateLine(const TInstant& requestedTs, TLineCallback func) const {
        Impl->IterateLine(requestedTs, std::move(func));
    }

    void TProbeSliceDumper::IterateDatacenter(const TInstant& requestedTs, TDatacenterCallback func) const {
        Impl->IterateDatacenter(requestedTs, std::move(func));
    }

    void TProbeSliceDumper::IterateSwitch(const TInstant& since, const TInstant& until, TSwitchCallback func) const {
        Impl->IterateSwitch(since, until, std::move(func));
    }

    void TProbeSliceDumper::IterateLine(const TInstant& since, const TInstant& until, TLineCallback func) const {
        Impl->IterateLine(since, until, std::move(func));
    }

    void TProbeSliceDumper::IterateDatacenter(const TInstant& since, const TInstant& until, TDatacenterCallback func) const {
        Impl->IterateDatacenter(since, until, std::move(func));
    }

    TThreadPool::TFuture TProbeSliceDumper::WaitSwitchTask() {
        return Impl->WaitSwitchTask();
    }

    TThreadPool::TFuture TProbeSliceDumper::WaitLineTask() {
        return Impl->WaitLineTask();
    }

    TThreadPool::TFuture TProbeSliceDumper::WaitDatacenterTask() {
        return Impl->WaitDatacenterTask();
    }
}
