#include <infra/netmon/probe_slice_dumper.h>
#include <infra/netmon/common_ut.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/tempdir.h>

class TProbeSliceDumperTest: public TTestBase {
    UNIT_TEST_SUITE(TProbeSliceDumperTest);
    UNIT_TEST(TestDumpSwitch)
    UNIT_TEST(TestDumpLine)
    UNIT_TEST(TestDumpDatacenter)
    UNIT_TEST_SUITE_END();

public:
    TProbeSliceDumperTest()
        : Now(TInstant::Now())
        , SourceHost(FindHost("jmon-test.search.yandex.net"))
        , TargetHost(FindHost("man1-0015.search.yandex.net"))
    {
        TSettings::Get()->SetHistoryStatePath(StateDirectory.Path());
    }

private:
    inline void TestDumpSwitch() {
        // check read
        TProbeSliceDumper dumper(GetTopologyStorage(), Merger);
        const auto previousTimestamp(Now - TSettings::Get()->GetSwitchAggregationInterval());
        const TLinePairKey pairKey(TargetHost.GetLine(), SourceHost.GetLine());
        Merger.SetTimestamp(previousTimestamp);
        WaitForFuture(dumper.WaitSwitchTask());
        Merger.SetTimestamp(Now);
        const auto result(dumper.Read(previousTimestamp, Merger.GetKey(), pairKey));
        UNIT_ASSERT(result);
        UNIT_ASSERT_VALUES_EQUAL(result->GetGenerated(), previousTimestamp);
        UNIT_ASSERT(!dumper.Read(Now - TSettings::Get()->GetSwitchAggregationInterval() * 2, Merger.GetKey(), pairKey));
        UNIT_ASSERT(dumper.Read(Now, Merger.GetKey(), pairKey));

        // check iteration
        bool found = false;
        dumper.IterateSwitch(previousTimestamp, [this, &found](auto key, auto index) {
            UNIT_ASSERT_VALUES_EQUAL(key.GetAggregatorKey(), Merger.GetKey());
            UNIT_ASSERT(index);
            found = true;
        });
        UNIT_ASSERT(found);
    }

    inline void TestDumpLine() {
        // check read
        TProbeSliceDumper dumper(GetTopologyStorage(), Merger);
        const auto previousTimestamp(Now - TSettings::Get()->GetLineAggregationInterval());
        const TDatacenterPairKey pairKey(TargetHost.GetDatacenter(), SourceHost.GetDatacenter());
        Merger.SetTimestamp(previousTimestamp);
        WaitForFuture(dumper.WaitLineTask());
        Merger.SetTimestamp(Now);
        const auto result(dumper.Read(previousTimestamp, Merger.GetKey(), pairKey));
        UNIT_ASSERT(result);
        UNIT_ASSERT_VALUES_EQUAL(result->GetGenerated(), previousTimestamp);
        UNIT_ASSERT(!dumper.Read(Now - TSettings::Get()->GetLineAggregationInterval() * 2, Merger.GetKey(), pairKey));
        UNIT_ASSERT(dumper.Read(Now, Merger.GetKey(), pairKey));

        // check iteration
        bool found = false;
        dumper.IterateLine(previousTimestamp, [this, &found](auto key, auto index) {
            UNIT_ASSERT_VALUES_EQUAL(key.GetAggregatorKey(), Merger.GetKey());
            UNIT_ASSERT(index);
            found = true;
        });
        UNIT_ASSERT(found);
    }

    inline void TestDumpDatacenter() {
        TProbeSliceDumper dumper(GetTopologyStorage(), Merger);
        const auto previousTimestamp(Now - TSettings::Get()->GetDcAggregationInterval());
        Merger.SetTimestamp(previousTimestamp);
        WaitForFuture(dumper.WaitDatacenterTask());
        Merger.SetTimestamp(Now);
        const auto result(dumper.Read(previousTimestamp, Merger.GetKey()));
        UNIT_ASSERT(result);
        UNIT_ASSERT_VALUES_EQUAL(result->GetGenerated(), previousTimestamp);
        UNIT_ASSERT(!dumper.Read(Now - TSettings::Get()->GetDcAggregationInterval() * 2, Merger.GetKey()));
        UNIT_ASSERT(dumper.Read(Now, Merger.GetKey()));

        // check iteration
        bool found = false;
        dumper.IterateDatacenter(previousTimestamp, [this, &found](auto key, auto index) {
            UNIT_ASSERT_VALUES_EQUAL(key.GetAggregatorKey(), Merger.GetKey());
            UNIT_ASSERT(index);
            found = true;
        });
        UNIT_ASSERT(found);
    }

    TTempDir StateDirectory;
    const TInstant Now;
    const TTopology::THostRef SourceHost;
    const TTopology::THostRef TargetHost;
    TFakeProbeSliceMerger Merger;
};

UNIT_TEST_SUITE_REGISTRATION(TProbeSliceDumperTest);
