#include <infra/netmon/probe_slice_history.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/history_placement.h>
#include <infra/netmon/state_history_keys.h>

namespace NNetmon {
    namespace {
        template <class TConfig>
        class TIndicesBuffer {
        public:
            using TIndexMap = typename TConfig::TIndexMap;

            inline typename TIndexMap::TRef GetCurrentOrMoveForward(const TConfig& config, const TInstant& now) {
                if (NextDumpTime <= now) {
                    MoveForward(config, now);
                }
                return Current;
            }

            inline typename TIndexMap::TRef GetPreviousOrDrop(const TInstant& timestamp) {
                if (!Previous) {
                    return nullptr;
                } else if (Previous->GetGenerated() <= timestamp) {
                    Previous.Reset();
                    return nullptr;
                }
                return Previous;
            }

            bool HasPrevious() const {
                return !!Previous;
            }

            typename TIndexMap::TRef GetIndexMap(const TConfig& config, const TInstant& requestedTs) const {
                if (Current && Current->GetGenerated() == config.RoundToInterval(requestedTs)) {
                    return Current;
                } else if (Previous && Previous->GetGenerated() == config.RoundToInterval(requestedTs)) {
                    return Previous;
                } else {
                    return nullptr;
                }
            }

        private:
            void MoveForward(const TConfig& config, const TInstant& now) {
                Previous.Reset(MakeAtomicShared<TIndexMap>(config.RoundToInterval(now)));
                Current.Swap(Previous);
                NextDumpTime = Current->GetGenerated() + config.GetChunkDuration();
            }

            typename TIndexMap::TRef Current;
            typename TIndexMap::TRef Previous;

            TInstant NextDumpTime;
        };

        template <class TIndicesBox>
        inline std::tuple<bool, TInstant> FillBuilderImpl(TIndicesBox& indices,
                                                          NNetmon::THistoryChunkBuilder& builder,
                                                          const TInstant& timestamp) {
            const auto indexMap(indices.Own()->GetPreviousOrDrop(timestamp));
            if (indexMap) {
                bool dumped(false);
                flatbuffers::FlatBufferBuilder flatBuilder;
                for (const auto& pair : *indexMap) {
                    const auto& index(*pair.second);
                    for (auto it(index.GetTree().Begin()); it != index.GetTree().End(); ++it) {
                        flatBuilder.Clear();
                        flatBuilder.Finish(it->ToProto(flatBuilder));
                        const auto key(ToHistoryKey(pair.first, it->GetKey()).AsTuple());
                        builder.Add(key, ToStringBuf(flatBuilder));
                        dumped = true;
                    }
                }
                return {dumped, indexMap->GetGenerated()};
            }
            return {false, TInstant::Zero()};
        }

        template <class TConfig>
        using TIndicesBox = TPlainLockedBox<TIndicesBuffer<TConfig>>;

        class TSwitchConfig {
        public:
            using THistoryKey = THistorySwitchPairKey;
            using TIndexMap = TSwitchIndexHistoryMap;
            using TIndicesBox = TIndicesBox<TSwitchConfig>;
            using TMaintainer = THistoryMaintainer<TSwitchConfig>;

            using TRealtimeIndex = TIndexMap::TRealtimeIndex;
            using TIncomingMap = THashMap<TProbeAggregatorKey, TRealtimeIndex::TRef>;
            using TIncomingBox = TPlainLockedBox<TIncomingMap>;
            using TEventHub = TEventHub<TRealtimeIndex::TRef>;

            TSwitchConfig(const IProbeSliceMerger& merger, const IProbeSliceDumper& dumper, TIndicesBox& indices)
                : Merger(merger)
                , Dumper(dumper)
                , Indices(indices)
            {
            }

            inline TStringBuf GetCodec() const {
                return TStringBuf("snappy");
            }

            inline ENetmonSignals UpdateMetric() const {
                return ENetmonSignals::SeriesDumperSwitchUpdateTime;
            }

            inline ENetmonSignals ReadMetric() const {
                return ENetmonSignals::SeriesDumperSwitchReadTime;
            }

            inline ENetmonSignals DumpMetric() const {
                return ENetmonSignals::SeriesDumperSwitchWriteTime;
            }

            TDuration GetDumpInterval() const {
                return TSettings::Get()->GetSwitchAggregationInterval();
            }

            TDuration GetChunkDuration() const {
                return GetDumpInterval() * 30;
            }

            TInstant RoundToInterval(const TInstant& ts) const {
                return RoundInstant(ts, GetChunkDuration());
            }

            inline TDuration GetTimeToLive() const {
                return TSettings::Get()->GetSwitchHistoryTimeToLive();
            }

            inline std::size_t GetBlobCacheSize() const {
                return 50;
            }

            inline TFsPath GetRoot() const {
                return TSettings::Get()->GetHistorySeriesPath().Child("switch");
            }

            const TEventHub& OnUpdated() const {
                return Merger.OnSwitchUpdated();
            }

            std::tuple<bool, TInstant> FillBuilder(NNetmon::THistoryChunkBuilder& builder, const TInstant& timestamp) const {
                return FillBuilderImpl(Indices, builder, timestamp);
            }

            bool HasAnythingToDump() const {
                return Indices.Own()->HasPrevious();
            }

            void IterateOverDumps(const TInstant& since, const TInstant& until, IProbeSliceDumper::TSwitchCallback func) const {
                Dumper.IterateSwitch(since, until, std::move(func));
            }

            TSwitchPairStateHistory::TRef FromProto(const TSwitchPairKey& pairKey, const TStringBuf& payload) const {
                return TSwitchPairStateHistory::Make(
                    pairKey.GetTarget(),
                    pairKey.GetSource(),
                    *flatbuffers::GetRoot<TSwitchPairStateHistory::TFlat>(payload.data())
                );
            }

        private:
            const IProbeSliceMerger& Merger;
            const IProbeSliceDumper& Dumper;
            TIndicesBox& Indices;
        };

        class TLineConfig {
        public:
            using THistoryKey = THistoryLinePairKey;
            using TIndexMap = TLineIndexHistoryMap;
            using TIndicesBox = TIndicesBox<TLineConfig>;
            using TMaintainer = THistoryMaintainer<TLineConfig>;

            using TRealtimeIndex = TIndexMap::TRealtimeIndex;
            using TIncomingMap = THashMap<TProbeAggregatorKey, TRealtimeIndex::TRef>;
            using TIncomingBox = TPlainLockedBox<TIncomingMap>;
            using TEventHub = TEventHub<TRealtimeIndex::TRef>;

            TLineConfig(const IProbeSliceMerger& merger, const IProbeSliceDumper& dumper, TIndicesBox& indices)
                : Merger(merger)
                , Dumper(dumper)
                , Indices(indices)
            {
            }

            inline TStringBuf GetCodec() const {
                return TStringBuf("snappy");
            }

            inline ENetmonSignals UpdateMetric() const {
                return ENetmonSignals::SeriesDumperQueueUpdateTime;
            }

            inline ENetmonSignals ReadMetric() const {
                return ENetmonSignals::SeriesDumperQueueReadTime;
            }

            inline ENetmonSignals DumpMetric() const {
                return ENetmonSignals::SeriesDumperQueueWriteTime;
            }

            TDuration GetDumpInterval() const {
                return TSettings::Get()->GetLineAggregationInterval();
            }

            TDuration GetChunkDuration() const {
                return GetDumpInterval() * 240;
            }

            TInstant RoundToInterval(const TInstant& ts) const {
                return RoundInstant(ts, GetChunkDuration());
            }

            inline TDuration GetTimeToLive() const {
                return TSettings::Get()->GetQueueHistoryTimeToLive();
            }

            inline std::size_t GetBlobCacheSize() const {
                return 100;
            }

            inline TFsPath GetRoot() const {
                return TSettings::Get()->GetHistorySeriesPath().Child("line");
            }

            const TEventHub& OnUpdated() const {
                return Merger.OnLineUpdated();
            }

            std::tuple<bool, TInstant> FillBuilder(NNetmon::THistoryChunkBuilder& builder, const TInstant& timestamp) const {
                return FillBuilderImpl(Indices, builder, timestamp);
            }

            bool HasAnythingToDump() const {
                return Indices.Own()->HasPrevious();
            }

            void IterateOverDumps(const TInstant& since, const TInstant& until, IProbeSliceDumper::TLineCallback func) const {
                Dumper.IterateLine(since, until, std::move(func));
            }

            TLinePairStateHistory::TRef FromProto(const TLinePairKey& pairKey, const TStringBuf& payload) const {
                return TLinePairStateHistory::Make(
                    pairKey.GetTarget(),
                    pairKey.GetSource(),
                    *flatbuffers::GetRoot<TLinePairStateHistory::TFlat>(payload.data())
                );
            }

        private:
            const IProbeSliceMerger& Merger;
            const IProbeSliceDumper& Dumper;
            TIndicesBox& Indices;
        };

        class TDatacenterConfig {
        public:
            using THistoryKey = THistoryDatacenterPairKey;
            using TIndexMap = TDatacenterIndexHistoryMap;
            using TIndicesBox = TIndicesBox<TDatacenterConfig>;
            using TMaintainer = THistoryMaintainer<TDatacenterConfig>;

            using TRealtimeIndex = TIndexMap::TRealtimeIndex;
            using TIncomingMap = THashMap<TProbeAggregatorKey, TRealtimeIndex::TRef>;
            using TIncomingBox = TPlainLockedBox<TIncomingMap>;
            using TEventHub = TEventHub<TRealtimeIndex::TRef>;

            TDatacenterConfig(const IProbeSliceMerger& merger, const IProbeSliceDumper& dumper, TIndicesBox& indices)
                : Merger(merger)
                , Dumper(dumper)
                , Indices(indices)
            {
            }

            inline TStringBuf GetCodec() const {
                return TStringBuf("snappy");
            }

            inline ENetmonSignals UpdateMetric() const {
                return ENetmonSignals::SeriesDumperDcUpdateTime;
            }

            inline ENetmonSignals ReadMetric() const {
                return ENetmonSignals::SeriesDumperDcReadTime;
            }

            inline ENetmonSignals DumpMetric() const {
                return ENetmonSignals::SeriesDumperDcWriteTime;
            }

            TDuration GetDumpInterval() const {
                return TSettings::Get()->GetDcAggregationInterval();
            }

            TDuration GetChunkDuration() const {
                return GetDumpInterval() * 720;
            }

            TInstant RoundToInterval(const TInstant& ts) const {
                return RoundInstant(ts, GetChunkDuration());
            }

            inline TDuration GetTimeToLive() const {
                return TSettings::Get()->GetDcHistoryTimeToLive();
            }

            inline std::size_t GetBlobCacheSize() const {
                return 200;
            }

            inline TFsPath GetRoot() const {
                return TSettings::Get()->GetHistorySeriesPath().Child("dc");
            }

            const TEventHub& OnUpdated() const {
                return Merger.OnDatacenterUpdated();
            }

            std::tuple<bool, TInstant> FillBuilder(NNetmon::THistoryChunkBuilder& builder, const TInstant& timestamp) const {
                return FillBuilderImpl(Indices, builder, timestamp);
            }

            void IterateOverDumps(const TInstant& since, const TInstant& until, IProbeSliceDumper::TDatacenterCallback func) const {
                Dumper.IterateDatacenter(since, until, std::move(func));
            }

            bool HasAnythingToDump() const {
                return Indices.Own()->HasPrevious();
            }

            TDatacenterPairStateHistory::TRef FromProto(const TDatacenterPairKey& pairKey, const TStringBuf& payload) const {
                return TDatacenterPairStateHistory::Make(
                    pairKey.GetTarget(),
                    pairKey.GetSource(),
                    *flatbuffers::GetRoot<TDatacenterPairStateHistory::TFlat>(payload.data())
                );
            }

        private:
            const IProbeSliceMerger& Merger;
            const IProbeSliceDumper& Dumper;
            TIndicesBox& Indices;
        };

        template <class TConfig>
        class TInMemoryHistoryUpdater : public TScheduledTask {
        public:
            using TSelf = TInMemoryHistoryUpdater<TConfig>;
            using TIndicesBox = typename TConfig::TIndicesBox;

            TInMemoryHistoryUpdater(const TConfig& config,
                                    typename TConfig::TMaintainer& maintainer,
                                    TIndicesBox& indices)
                : TScheduledTask(config.GetDumpInterval(), true)
                , Config(config)
                , Maintainer(maintainer)
                , Indices(indices)
                , Executor("MemoryHistoryUpdater")
                , Preloaded(false)
            {
            }

            TThreadPool::TFuture Run() override {
                return Executor.Add([this]() {
                    if (!Preloaded) {
                        Preloaded = true;
                        PreloadData().GetValueSync();
                    }
                    TUnistatTimer timer{TUnistat::Instance(), Config.UpdateMetric()};
                    DoRun().GetValueSync();
                });
            }

            typename TConfig::TEventHub::TSubscriptionGuard Subscribe() {
                return Config.OnUpdated().Subscribe([this](typename TConfig::TRealtimeIndex::TRef index) {
                    IncomingBox.Own()->emplace(index->GetKey(), index);
                    Spin();
                });
            }

            bool IsReady() const noexcept {
                const auto lastUpdateTime(*LastUpdateTime.Own());
                if (lastUpdateTime) {
                    return lastUpdateTime + GetInterval() * 3 > TInstant::Now();
                }
                return true;
            }

        private:
            TThreadPool::TFuture PreloadData() {
                TSimpleTimer timer;
                const auto since(Config.RoundToInterval(TInstant::Now()));
                const auto until(since + Config.GetChunkDuration());
                TInstant timestamp;
                TVector<TThreadPool::TFuture> futures;
                Config.IterateOverDumps(since, until, [this, since, until, &futures, &timestamp](auto, auto index) {
                    if (since <= index->GetGenerated() && index->GetGenerated() < until && TInstant::Now() < until) {
                        const auto current(Indices.Own()->GetCurrentOrMoveForward(Config, index->GetGenerated()));
                        futures.emplace_back(Executor.Add([current, index]() {
                            current->Append(*index, false);
                        }));
                        if (index->GetGenerated() != timestamp) {
                            NThreading::WaitExceptionOrAll(futures).GetValueSync();
                            futures.clear();
                            timestamp = index->GetGenerated();
                        }
                    }
                });
                INFO_LOG << "History preloading for " << TypeName(*this) << " took " << timer.Get() << Endl;
                return NThreading::WaitExceptionOrAll(futures);
            }

            TThreadPool::TFuture DoRun() {
                TVector<typename TConfig::TRealtimeIndex::TRef> foundIndices;
                {
                    const auto incoming(IncomingBox.Own());
                    foundIndices.reserve(incoming->size());
                    for (auto& pair : *incoming) {
                        foundIndices.emplace_back(std::move(pair.second));
                    }
                    incoming->clear();
                }

                if (foundIndices.empty()) {
                    return NThreading::MakeFuture();
                }

                // TODO: some points may go to the wrong chunk, deal with it
                SortBy(foundIndices.begin(), foundIndices.end(), [](const auto& x) {
                    return x->GetGenerated();
                });

                TInstant timestamp;
                TVector<TThreadPool::TFuture> futures;
                for (const auto& index : foundIndices) {
                    timestamp = Max(timestamp, index->GetGenerated());
                    const auto current(Indices.Own()->GetCurrentOrMoveForward(Config, index->GetGenerated()));
                    futures.emplace_back(Executor.Add([current, index]() {
                        current->Append(*index);
                    }));
                }

                return NThreading::WaitExceptionOrAll(futures).Apply([this, timestamp](const TThreadPool::TFuture& future_) {
                    future_.GetValue();
                    // start dumping
                    Maintainer.SpinDumper();
                    *LastUpdateTime.Own() = timestamp;
                });
            }

            const TConfig& Config;
            typename TConfig::TMaintainer& Maintainer;
            TIndicesBox& Indices;
            typename TConfig::TIncomingBox IncomingBox;
            TBaseThreadExecutor Executor;
            TPlainLockedBox<TInstant> LastUpdateTime;
            bool Preloaded;
        };

        using TSwitchUpdater = TInMemoryHistoryUpdater<TSwitchConfig>;
        using TLineUpdater = TInMemoryHistoryUpdater<TLineConfig>;
        using TDatacenterUpdater = TInMemoryHistoryUpdater<TDatacenterConfig>;
    }

    class TProbeSliceHistory::TImpl {
    public:
        TImpl(const IProbeSliceMerger& merger, const IProbeSliceDumper& dumper)
            : SwitchConfig(merger, dumper, SwitchIndices)
            , LineConfig(merger, dumper, LineIndices)
            , DatacenterConfig(merger, dumper, DatacenterIndices)

            , SwitchMaintainer(SwitchConfig)
            , LineMaintainer(LineConfig)
            , DatacenterMaintainer(DatacenterConfig)

            , SwitchUpdater(SwitchConfig, SwitchMaintainer, SwitchIndices)
            , LineUpdater(LineConfig, LineMaintainer, LineIndices)
            , DatacenterUpdater(DatacenterConfig, DatacenterMaintainer, DatacenterIndices)

            , SwitchUpdaterGuard(SwitchUpdater.Schedule())
            , LineUpdaterGuard(LineUpdater.Schedule())
            , DatacenterUpdaterGuard(DatacenterUpdater.Schedule())

            , SwitchSubscription(SwitchUpdater.Subscribe())
            , LineSubscription(LineUpdater.Subscribe())
            , DatacenterSubscription(DatacenterUpdater.Subscribe())
        {
        }

        inline bool IsSwitchReady() const noexcept {
            return SwitchUpdater.IsReady() && SwitchMaintainer.IsReady();
        }

        inline bool IsLineReady() const noexcept {
            return LineUpdater.IsReady() && LineMaintainer.IsReady();
        }

        inline bool IsDatacenterReady() const noexcept {
            return DatacenterUpdater.IsReady() && DatacenterMaintainer.IsReady();
        }

        TSwitchPairStateHistory::TRef Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TSwitchPairKey& pairKey) const {
            return RealtimeAwareRead(SwitchConfig, SwitchIndices, SwitchMaintainer, requestedTs, key, pairKey);
        }

        TLinePairStateHistory::TRef Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
            return RealtimeAwareRead(LineConfig, LineIndices, LineMaintainer, requestedTs, key, pairKey);
        }

        TDatacenterPairStateHistory::TRef Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
            return RealtimeAwareRead(DatacenterConfig, DatacenterIndices, DatacenterMaintainer, requestedTs, key, pairKey);
        }

        TVector<TSwitchPairStateHistory::TRef> Read(
                const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TSwitchPairKey& pairKey) const {
            return RangeRead(SwitchConfig, since, until, key, pairKey);
        }

        TVector<TLinePairStateHistory::TRef> Read(
                const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
            return RangeRead(LineConfig, since, until, key, pairKey);
        }

        TVector<TDatacenterPairStateHistory::TRef> Read(
                const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
            return RangeRead(DatacenterConfig, since, until, key, pairKey);
        }

        TDuration GetSwitchChunkDuration() const {
            return SwitchConfig.GetChunkDuration();
        }

        TDuration GetLineChunkDuration() const {
            return LineConfig.GetChunkDuration();
        }

        TDuration GetDatacenterChunkDuration() const {
            return DatacenterConfig.GetChunkDuration();
        }

        TThreadPool::TFuture WaitSwitchUpdate() {
            return SwitchUpdater.SpinAndWait();
        }

        TThreadPool::TFuture WaitLineUpdate() {
            return LineUpdater.SpinAndWait();
        }

        TThreadPool::TFuture WaitDatacenterUpdate() {
            return DatacenterUpdater.SpinAndWait();
        }

        TThreadPool::TFuture WaitSwitchDump() {
            return SwitchMaintainer.WaitDumper();
        }

        TThreadPool::TFuture WaitLineDump() {
            return LineMaintainer.WaitDumper();
        }

        TThreadPool::TFuture WaitDatacenterDump() {
            return DatacenterMaintainer.WaitDumper();
        }

    private:
        template <class TConfig, class TIndices, class TMaintainer, class TPairKey>
        inline typename TConfig::TIndexMap::TIndex::TState::TRef RealtimeAwareRead(
                const TConfig& config, const TIndices& indices, const TMaintainer& maintainer,
                const TInstant& requestedTs, const TProbeAggregatorKey& key, const TPairKey& pairKey) const {
            if (!pairKey.IsValid()) {
                return nullptr;
            }

            const auto indexMap(indices.Own()->GetIndexMap(config, requestedTs));
            if (indexMap) {
                return indexMap->FindState(key, pairKey);
            }

            const auto historyKey(ToHistoryKey(key, pairKey).AsTuple());
            TBuffer buffer;
            TStringBuf payload;
            if (maintainer.Read(historyKey, requestedTs, payload, buffer)) {
                return config.FromProto(pairKey, payload);
            } else {
                return nullptr;
            }
        }

        template <class TConfig, class TPairKey>
        TVector<typename TConfig::TIndexMap::TIndex::TState::TRef> RangeRead(
                const TConfig& config, const TInstant& since, const TInstant& until,
                const TProbeAggregatorKey& key, const TPairKey& pairKey) const {
            TVector<typename TConfig::TIndexMap::TIndex::TState::TRef> result;
            auto epoch(config.RoundToInterval(since));
            while (epoch <= config.RoundToInterval(until)) {
                auto state(Read(epoch, key, pairKey));
                if (state) {
                    result.emplace_back(std::move(state));
                }
                epoch += config.GetChunkDuration();
            }
            return result;
        }

        TSwitchConfig::TIndicesBox SwitchIndices;
        TLineConfig::TIndicesBox LineIndices;
        TDatacenterConfig::TIndicesBox DatacenterIndices;

        TSwitchConfig SwitchConfig;
        TLineConfig LineConfig;
        TDatacenterConfig DatacenterConfig;

        TSwitchConfig::TMaintainer SwitchMaintainer;
        TLineConfig::TMaintainer LineMaintainer;
        TDatacenterConfig::TMaintainer DatacenterMaintainer;

        TSwitchUpdater SwitchUpdater;
        TLineUpdater LineUpdater;
        TDatacenterUpdater DatacenterUpdater;

        TScheduledTask::TTaskGuard SwitchUpdaterGuard;
        TScheduledTask::TTaskGuard LineUpdaterGuard;
        TScheduledTask::TTaskGuard DatacenterUpdaterGuard;

        TSwitchConfig::TEventHub::TSubscriptionGuard SwitchSubscription;
        TLineConfig::TEventHub::TSubscriptionGuard LineSubscription;
        TDatacenterConfig::TEventHub::TSubscriptionGuard DatacenterSubscription;
    };

    TProbeSliceHistory::TProbeSliceHistory(const IProbeSliceMerger& merger, const IProbeSliceDumper& dumper)
        : Impl(MakeHolder<TImpl>(merger, dumper))
    {
    }

    TProbeSliceHistory::~TProbeSliceHistory() {
    }

    bool TProbeSliceHistory::IsReady() const noexcept {
        return Impl->IsSwitchReady() && Impl->IsLineReady() && Impl->IsDatacenterReady();
    }

    bool TProbeSliceHistory::IsSwitchReady() const noexcept {
        return Impl->IsSwitchReady();
    }

    bool TProbeSliceHistory::IsLineReady() const noexcept {
        return Impl->IsLineReady();
    }

    bool TProbeSliceHistory::IsDatacenterReady() const noexcept {
        return Impl->IsDatacenterReady();
    }

    TSwitchPairStateHistory::TRef TProbeSliceHistory::Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TSwitchPairKey& pairKey) const {
        return Impl->Read(requestedTs, key, pairKey);
    }

    TLinePairStateHistory::TRef TProbeSliceHistory::Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
        return Impl->Read(requestedTs, key, pairKey);
    }

    TDatacenterPairStateHistory::TRef TProbeSliceHistory::Read(const TInstant& requestedTs, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
        return Impl->Read(requestedTs, key, pairKey);
    }

    TVector<TSwitchPairStateHistory::TRef> TProbeSliceHistory::Read(
            const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TSwitchPairKey& pairKey) const {
        return Impl->Read(since, until, key, pairKey);
    }

    TVector<TLinePairStateHistory::TRef> TProbeSliceHistory::Read(
            const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TLinePairKey& pairKey) const {
        return Impl->Read(since, until, key, pairKey);
    }

    TVector<TDatacenterPairStateHistory::TRef> TProbeSliceHistory::Read(
            const TInstant& since, const TInstant& until, const TProbeAggregatorKey& key, const TDatacenterPairKey& pairKey) const {
        return Impl->Read(since, until, key, pairKey);
    }

    TDuration TProbeSliceHistory::GetSwitchChunkDuration() const {
        return Impl->GetSwitchChunkDuration();
    }

    TDuration TProbeSliceHistory::GetLineChunkDuration() const {
        return Impl->GetLineChunkDuration();
    }

    TDuration TProbeSliceHistory::GetDatacenterChunkDuration() const {
        return Impl->GetDatacenterChunkDuration();
    }

    TThreadPool::TFuture TProbeSliceHistory::WaitSwitchUpdate() {
        return Impl->WaitSwitchUpdate();
    }

    TThreadPool::TFuture TProbeSliceHistory::WaitLineUpdate() {
        return Impl->WaitLineUpdate();
    }

    TThreadPool::TFuture TProbeSliceHistory::WaitDatacenterUpdate() {
        return Impl->WaitDatacenterUpdate();
    }

    TThreadPool::TFuture TProbeSliceHistory::WaitSwitchDump() {
        return Impl->WaitSwitchDump();
    }

    TThreadPool::TFuture TProbeSliceHistory::WaitLineDump() {
        return Impl->WaitLineDump();
    }

    TThreadPool::TFuture TProbeSliceHistory::WaitDatacenterDump() {
        return Impl->WaitDatacenterDump();
    }
}
