#include <infra/netmon/probe_slice_history.h>
#include <infra/netmon/common_ut.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/tempdir.h>

class TFakeProbeSliceDumper: public IProbeSliceDumper, public TNonCopyable {
public:
    TSwitchPairIndex::TRef Read(const TInstant&, const TProbeAggregatorKey&, const TLinePairKey&) const override {
        return nullptr;
    }
    TLinePairIndex::TRef Read(const TInstant&, const TProbeAggregatorKey&, const TDatacenterPairKey&) const override {
        return nullptr;
    }
    TDatacenterPairIndex::TRef Read(const TInstant&, const TProbeAggregatorKey&) const override {
        return nullptr;
    }

    void IterateSwitch(const TInstant&, TSwitchCallback) const override {
    }
    void IterateLine(const TInstant&, TLineCallback) const override {
    }
    void IterateDatacenter(const TInstant&, TDatacenterCallback) const override {
    }

    void IterateSwitch(const TInstant&, const TInstant&, TSwitchCallback) const override {
    }
    void IterateLine(const TInstant&, const TInstant&, TLineCallback) const override {
    }
    void IterateDatacenter(const TInstant&, const TInstant&, TDatacenterCallback) const override {
    }
};

class TProbeSliceHistoryTest: public TTestBase {
    UNIT_TEST_SUITE(TProbeSliceHistoryTest);
    UNIT_TEST(TestUpdateSwitch)
    UNIT_TEST(TestUpdateLine)
    UNIT_TEST(TestUpdateDatacenter)
    UNIT_TEST_SUITE_END();

public:
    TProbeSliceHistoryTest()
        : Now(TInstant::Now())
        , SourceHost(FindHost("jmon-test.search.yandex.net"))
        , TargetHost(FindHost("man1-0015.search.yandex.net"))
    {
        TSettings::Get()->SetHistorySeriesPath(StateDirectory.Path());
    }

private:
    inline void TestUpdateSwitch() {
        TProbeSliceHistory history(Merger, Dumper);

        const auto previousTimestamp(Now - history.GetSwitchChunkDuration());
        const TSwitchPairKey pairKey(TargetHost.GetSwitch(), SourceHost.GetSwitch());
        Merger.SetTimestamp(previousTimestamp);
        Merger.NotifySwitch();
        WaitForFuture(history.WaitSwitchUpdate());

        const auto result(history.Read(previousTimestamp, Merger.GetKey(), pairKey));
        UNIT_ASSERT(result);

        Merger.SetTimestamp(Now);
        Merger.NotifySwitch();
        WaitForFuture(history.WaitSwitchUpdate());
        WaitForFuture(history.WaitSwitchDump());
        UNIT_ASSERT(!history.Read(previousTimestamp - history.GetSwitchChunkDuration(), Merger.GetKey(), pairKey));
        UNIT_ASSERT(history.Read(previousTimestamp, Merger.GetKey(), pairKey));

        Merger.SetTimestamp(Now + history.GetSwitchChunkDuration());
        Merger.NotifySwitch();
        WaitForFuture(history.WaitSwitchUpdate());
        WaitForFuture(history.WaitSwitchDump());
        UNIT_ASSERT(history.Read(previousTimestamp, Merger.GetKey(), pairKey));
    }

    inline void TestUpdateLine() {
        TProbeSliceHistory history(Merger, Dumper);

        const auto previousTimestamp(Now - history.GetLineChunkDuration());
        const TLinePairKey pairKey(TargetHost.GetLine(), SourceHost.GetLine());
        Merger.SetTimestamp(previousTimestamp);
        Merger.NotifyLine();
        WaitForFuture(history.WaitLineUpdate());

        const auto result(history.Read(previousTimestamp, Merger.GetKey(), pairKey));
        UNIT_ASSERT(result);

        Merger.SetTimestamp(Now);
        Merger.NotifyLine();
        WaitForFuture(history.WaitLineUpdate());
        WaitForFuture(history.WaitLineDump());
        UNIT_ASSERT(!history.Read(previousTimestamp - history.GetLineChunkDuration(), Merger.GetKey(), pairKey));
        UNIT_ASSERT(history.Read(previousTimestamp, Merger.GetKey(), pairKey));

        Merger.SetTimestamp(Now + history.GetLineChunkDuration());
        Merger.NotifyLine();
        WaitForFuture(history.WaitLineUpdate());
        WaitForFuture(history.WaitLineDump());
        UNIT_ASSERT(history.Read(previousTimestamp, Merger.GetKey(), pairKey));
    }

    inline void TestUpdateDatacenter() {
        TProbeSliceHistory history(Merger, Dumper);

        const auto previousTimestamp(Now - history.GetDatacenterChunkDuration());
        const TDatacenterPairKey pairKey(TargetHost.GetDatacenter(), SourceHost.GetDatacenter());
        Merger.SetTimestamp(previousTimestamp);
        Merger.NotifyDatacenter();
        WaitForFuture(history.WaitDatacenterUpdate());

        const auto result(history.Read(previousTimestamp, Merger.GetKey(), pairKey));
        UNIT_ASSERT(result);

        Merger.SetTimestamp(Now);
        Merger.NotifyDatacenter();
        WaitForFuture(history.WaitDatacenterUpdate());
        WaitForFuture(history.WaitDatacenterDump());
        UNIT_ASSERT(!history.Read(previousTimestamp - history.GetDatacenterChunkDuration(), Merger.GetKey(), pairKey));
        UNIT_ASSERT(history.Read(previousTimestamp, Merger.GetKey(), pairKey));

        Merger.SetTimestamp(Now + history.GetDatacenterChunkDuration());
        Merger.NotifyDatacenter();
        WaitForFuture(history.WaitDatacenterUpdate());
        WaitForFuture(history.WaitDatacenterDump());
        UNIT_ASSERT(history.Read(previousTimestamp, Merger.GetKey(), pairKey));
    }

    TTempDir StateDirectory;
    const TInstant Now;
    const TTopology::THostRef SourceHost;
    const TTopology::THostRef TargetHost;
    TFakeProbeSliceMerger Merger;
    TFakeProbeSliceDumper Dumper;
};

UNIT_TEST_SUITE_REGISTRATION(TProbeSliceHistoryTest);
